// background.js — SuperPay Extension Service Worker (Safari)
// Handles: auth state, card caching, merchant sync, message passing to content scripts

if (typeof browser !== 'undefined' && typeof chrome === 'undefined') {
  globalThis.chrome = browser;
}

const API_BASE = 'https://superpayrewards.com';
const CACHE_TTL = 5 * 60 * 1000;
const MERCHANT_SYNC_TTL = 24 * 60 * 60 * 1000;

const notifiedMerchants = new Map();
const hasNotifications = typeof chrome !== 'undefined' && chrome.notifications && typeof chrome.notifications.create === 'function';

// ── Initialization ──

chrome.runtime.onInstalled.addListener(() => {
  chrome.storage.local.set({
    authToken: null,
    user: null,
    cards: [],
    goals: [],
    lastCardSync: 0,
    lastMerchantSync: 0,
    serverMerchants: null,
    settings: {
      overlayEnabled: true,
      overlayPosition: 'bottom-right',
      autoHideSeconds: 15,
      showSecondBest: true,
      showAiExplanation: true,
      notificationsEnabled: true,
      blacklistedDomains: [],
    },
  });

  // Set up periodic sync alarm
  chrome.alarms.create('syncCards', { periodInMinutes: 5 });
  chrome.alarms.create('syncMerchants', { periodInMinutes: 1440 }); // 24 hours
});

// ── Alarm Handlers ──

chrome.alarms.onAlarm.addListener(async (alarm) => {
  if (alarm.name === 'syncCards') {
    await syncCards();
  }
  if (alarm.name === 'syncMerchants') {
    await syncMerchantDatabase();
  }
});

// ── Tab Navigation Listener ──

chrome.tabs.onUpdated.addListener(async (tabId, changeInfo, tab) => {
  if (changeInfo.status !== 'complete' || !tab.url) return;
  if (tab.url.startsWith('chrome://') || tab.url.startsWith('chrome-extension://')) return;

  const data = await chrome.storage.local.get(['authToken', 'cards', 'settings']);
  if (!data.authToken || !data.settings?.overlayEnabled) return;

  // Check if domain is blacklisted
  try {
    const hostname = new URL(tab.url).hostname.replace(/^www\./, '');
    if (data.settings.blacklistedDomains?.includes(hostname)) return;
  } catch { return; }

  // Send page info to content script for checkout detection
  try {
    await chrome.tabs.sendMessage(tabId, {
      type: 'CHECK_PAGE',
      url: tab.url,
      cards: data.cards,
      settings: data.settings,
    });
  } catch {
    // Content script not ready yet — ignore
  }
});

// ── Message Handler (from popup and content scripts) ──

chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  handleMessage(message, sender).then(sendResponse).catch(err => {
    console.error('Message handler error:', err);
    sendResponse({ error: err.message });
  });
  return true; // Keep channel open for async response
});

async function handleMessage(message, sender) {
  switch (message.type) {
    case 'LOGIN':
      return await handleLogin(message.email, message.password);

    case 'LOGOUT':
      await chrome.storage.local.set({
        authToken: null,
        user: null,
        cards: [],
        goals: [],
        lastCardSync: 0,
      });
      return { success: true };

    case 'GET_AUTH_STATE':
      return await getAuthState();

    case 'GET_CARDS':
      return await getCards();

    case 'GET_RECOMMENDATION':
      return await getRecommendation(message.merchant, message.category, message.amount);

    case 'PARSE_URL':
      return await parseCheckoutUrl(message.url);

    case 'GET_AI_EXPLANATION':
      return await getAiExplanation(message.data);

    case 'LOG_TRANSACTION':
      return await logTransaction(message.transaction);

    case 'SEARCH_RETAILERS':
      return await searchRetailers(message.query);

    case 'SYNC_CARDS':
      return await syncCards(true);

    case 'UPDATE_GOALS':
      return await handleUpdateGoals(message.goals);

    case 'GET_MULTI_GOAL_RECOMMENDATION':
      return await getMultiGoalRecommendation(message.category, message.amount);

    case 'GET_ROADMAPS':
      return await getRoadmaps();

    case 'CHECKOUT_DETECTED':
      return await handleCheckoutDetected(message, sender);

    case 'MERCHANT_PAGE_DETECTED':
      return await handleMerchantPageDetected(message, sender);

    case 'UNKNOWN_MERCHANT_DETECTED':
      return await handleUnknownMerchantDetected(message, sender);

    default:
      return { error: 'Unknown message type' };
  }
}

// ── Auth ──

async function handleLogin(email, password) {
  try {
    const res = await fetch(`${API_BASE}/api/auth/login`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ email, password }),
    });

    if (!res.ok) {
      const err = await res.json().catch(() => ({}));
      return { success: false, error: err.message || 'Login failed' };
    }

    const data = await res.json();
    await chrome.storage.local.set({
      authToken: data.token,
      user: data.user,
      goals: data.user?.goals || [],
      subscriptionTier: data.user?.subscriptionTier || '',
    });

    // Immediately sync cards and merchants after login
    await syncCards(true);
    await syncMerchantDatabase();

    return { success: true, user: data.user };
  } catch (err) {
    return { success: false, error: err.message };
  }
}

async function getAuthState() {
  const data = await chrome.storage.local.get(['authToken', 'user', 'goals']);
  if (!data.authToken) return { loggedIn: false };

  // Verify token is still valid
  try {
    const res = await fetch(`${API_BASE}/api/auth/me`, {
      headers: { 'Authorization': `Bearer ${data.authToken}` },
    });
    if (!res.ok) {
      await chrome.storage.local.set({ authToken: null, user: null });
      return { loggedIn: false };
    }
    const user = await res.json();
    await chrome.storage.local.set({ user, goals: user.goals || [], subscriptionTier: user.subscriptionTier || '' });
    return { loggedIn: true, user };
  } catch {
    return { loggedIn: true, user: data.user }; // Offline — use cached
  }
}

// ── Card Sync ──

async function syncCards(force = false) {
  const data = await chrome.storage.local.get(['authToken', 'lastCardSync']);
  if (!data.authToken) return { cards: [] };

  if (!force && data.lastCardSync && (Date.now() - data.lastCardSync < CACHE_TTL)) {
    const cached = await chrome.storage.local.get(['cards']);
    return { cards: cached.cards || [] };
  }

  try {
    const res = await fetch(`${API_BASE}/api/cards`, {
      headers: { 'Authorization': `Bearer ${data.authToken}` },
    });
    if (!res.ok) return { cards: [] };

    const cards = await res.json();
    await chrome.storage.local.set({ cards, lastCardSync: Date.now() });
    return { cards };
  } catch {
    const cached = await chrome.storage.local.get(['cards']);
    return { cards: cached.cards || [] };
  }
}

async function getCards() {
  const data = await chrome.storage.local.get(['cards', 'lastCardSync']);
  if (data.lastCardSync && (Date.now() - data.lastCardSync < CACHE_TTL)) {
    return { cards: data.cards || [] };
  }
  return await syncCards(true);
}

// ── Roadmaps ──

async function getRoadmaps() {
  const data = await chrome.storage.local.get(['authToken']);
  if (!data.authToken) return { roadmaps: [] };

  try {
    const res = await fetch(`${API_BASE}/api/roadmaps`, {
      headers: { 'Authorization': `Bearer ${data.authToken}` },
    });
    if (!res.ok) return { roadmaps: [] };
    const roadmaps = await res.json();
    return { roadmaps: Array.isArray(roadmaps) ? roadmaps : [] };
  } catch {
    return { roadmaps: [] };
  }
}

// ── Merchant Database Sync ──

async function syncMerchantDatabase() {
  const data = await chrome.storage.local.get(['authToken', 'lastMerchantSync']);
  if (!data.authToken) return;

  try {
    const res = await fetch(`${API_BASE}/api/extension/config`, {
      headers: { 'Authorization': `Bearer ${data.authToken}` },
    });
    if (!res.ok) return;

    const config = await res.json();
    if (config.merchants) {
      await chrome.storage.local.set({
        serverMerchants: config.merchants,
        lastMerchantSync: Date.now(),
      });
    }
  } catch {
    // Use local fallback — no problem
  }
}

// ── Recommendations ──

async function getRecommendation(merchant, category, amount) {
  const data = await chrome.storage.local.get(['cards', 'goals']);
  const cards = data.cards || [];
  const goals = data.goals || [];

  if (cards.length === 0) return { recommendation: null, reason: 'No cards added' };

  const activeCards = cards.filter(c => c.isActive);
  if (activeCards.length === 0) return { recommendation: null, reason: 'No active cards' };

  const calcAmount = amount || 100;
  const results = getOptimalCard(activeCards, category, calcAmount, goals);
  if (results.length === 0) return { recommendation: null };

  return {
    recommendation: results[0],
    allCards: results,
    merchant,
    category,
    amount: amount || null,
  };
}

async function parseCheckoutUrl(url) {
  const data = await chrome.storage.local.get(['authToken']);
  if (!data.authToken) return { found: false };

  try {
    const res = await fetch(`${API_BASE}/api/parse-checkout-url`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${data.authToken}`,
      },
      body: JSON.stringify({ url }),
    });
    if (!res.ok) return { found: false };
    return await res.json();
  } catch {
    return { found: false };
  }
}

async function handleCheckoutDetected(message, sender) {
  const { merchant, category, amount, url } = message;
  const realAmount = amount && amount > 0 ? amount : null;

  let result = await getRecommendation(merchant, category, realAmount);

  if (!result.recommendation && url) {
    const serverResult = await parseCheckoutUrl(url);
    if (serverResult.found && serverResult.bestCard) {
      const calcAmount = realAmount || 100;
      result = {
        recommendation: {
          card: serverResult.bestCard,
          rewardRate: serverResult.bestCard.rewardRate,
          rewardsEarned: calcAmount * serverResult.bestCard.rewardRate / 100,
          reason: serverResult.bestCard.reason,
          isBest: true,
        },
        merchant: serverResult.merchant,
        category: serverResult.category,
        amount: realAmount,
      };
    }
  }

  if (result.recommendation && sender.tab?.id) {
    const data = await chrome.storage.local.get(['settings', 'goals']);
    chrome.tabs.sendMessage(sender.tab.id, {
      type: 'SHOW_OVERLAY',
      recommendation: result.recommendation,
      allCards: result.allCards,
      merchant: result.merchant || merchant,
      category: result.category || category,
      amount: result.amount || realAmount,
      settings: data.settings,
      goals: data.goals || [],
    });
  }

  return result;
}

async function handleMerchantPageDetected(message, sender) {
  const { merchant, category, url } = message;

  const data = await chrome.storage.local.get(['authToken', 'cards', 'settings']);
  if (!data.authToken || !data.cards || data.cards.length === 0) return { notified: false };
  if (data.settings?.notificationsEnabled === false) return { notified: false };

  const cacheKey = merchant + '_' + (sender.tab?.id || '');
  const lastNotified = notifiedMerchants.get(cacheKey);
  const now = Date.now();
  if (lastNotified && (now - lastNotified) < 30 * 60 * 1000) return { notified: false };

  const result = await getRecommendation(merchant, category, null);
  if (!result.recommendation) return { notified: false };

  const card = result.recommendation.card || result.recommendation;
  const cardName = card.nickname || card.issuer || 'Your Card';
  const rewardType = card.rewardType === 'miles' ? 'miles' : card.rewardType === 'points' ? 'points' : 'cash back';
  const rate = result.recommendation.rewardRate;

  notifiedMerchants.set(cacheKey, now);

  if (hasNotifications) {
    try {
      chrome.notifications.create(`superpay-${Date.now()}`, {
        type: 'basic',
        iconUrl: chrome.runtime.getURL('icons/icon128.png'),
        title: `SuperPay: Use ${cardName}`,
        message: `${rate}x ${rewardType} at ${merchant}. Tap to see details.`,
        priority: 1,
      });
    } catch {}
  }

  return { notified: true, card: cardName, rate };
}

const discoveredMerchants = new Map();

async function handleUnknownMerchantDetected(message, sender) {
  const { merchant, category, confidence, url, domain } = message;

  const data = await chrome.storage.local.get(['authToken', 'cards', 'settings']);
  if (!data.authToken || !data.cards || data.cards.length === 0) return { notified: false };
  if (data.settings?.notificationsEnabled === false) return { notified: false };

  const cacheKey = domain + '_' + (sender.tab?.id || '');
  const lastNotified = notifiedMerchants.get(cacheKey);
  const now = Date.now();
  if (lastNotified && (now - lastNotified) < 30 * 60 * 1000) return { notified: false };

  let resolvedCategory = category;
  const cached = discoveredMerchants.get(domain);
  if (cached) {
    resolvedCategory = cached.category;
  } else {
    try {
      const res = await fetch(`${API_BASE}/api/categorize-merchant`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${data.authToken}`,
        },
        body: JSON.stringify({ merchantName: merchant }),
      });
      if (res.ok) {
        const catData = await res.json();
        if (catData.category) {
          resolvedCategory = catData.category;
        }
      }
    } catch {}
    discoveredMerchants.set(domain, { name: merchant, category: resolvedCategory });
  }

  const result = await getRecommendation(merchant, resolvedCategory, null);
  if (!result.recommendation) return { notified: false };

  const card = result.recommendation.card || result.recommendation;
  const cardName = card.nickname || card.issuer || 'Your Card';
  const rewardType = card.rewardType === 'miles' ? 'miles' : card.rewardType === 'points' ? 'points' : 'cash back';
  const rate = result.recommendation.rewardRate;

  notifiedMerchants.set(cacheKey, now);

  if (hasNotifications) {
    try {
      chrome.notifications.create(`superpay-${Date.now()}`, {
        type: 'basic',
        iconUrl: chrome.runtime.getURL('icons/icon128.png'),
        title: `SuperPay: Use ${cardName}`,
        message: `${rate}x ${rewardType} at ${merchant}. Tap to see details.`,
        priority: 1,
      });
    } catch {}
  }

  return { notified: true, card: cardName, rate, discovered: true };
}

if (hasNotifications) {
  chrome.notifications.onClicked.addListener((notificationId) => {
    if (notificationId.startsWith('superpay-')) {
      chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
        if (tabs[0]) {
          chrome.tabs.update(tabs[0].id, { active: true });
          chrome.windows.update(tabs[0].windowId, { focused: true });
        }
      });
      chrome.notifications.clear(notificationId);
    }
  });
}

// ── Goals ──

async function handleUpdateGoals(goals) {
  const stored = await chrome.storage.local.get(['authToken']);
  await chrome.storage.local.set({ goals });

  if (stored.authToken) {
    try {
      await fetch(`${API_BASE}/api/user/goals`, {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${stored.authToken}`,
        },
        body: JSON.stringify({ goals }),
      });
    } catch {
      // Offline — local storage already updated
    }
  }

  return { success: true, goals };
}

// ── Multi-Goal Recommendations ──

async function getMultiGoalRecommendation(category, amount) {
  const data = await chrome.storage.local.get(['cards', 'goals']);
  const cards = (data.cards || []).filter(c => c.isActive !== false);
  const userGoals = data.goals || [];

  if (cards.length === 0) return { results: [] };

  const ALL_GOALS = [
    { id: 'maximize_rewards', label: 'Max Rewards', icon: String.fromCodePoint(0x1F3C6), color: '#FFB800' },
    { id: 'build_credit', label: 'Protect Credit', icon: String.fromCodePoint(0x1F4C8), color: '#00C896' },
    { id: 'budget_conscious', label: 'Budget', icon: String.fromCodePoint(0x1F4B0), color: '#FF6B6B' },
    { id: 'travel_focus', label: 'Travel', icon: String.fromCodePoint(0x2708), color: '#A855F7' },
    { id: 'cashback_focus', label: 'Cash Back', icon: String.fromCodePoint(0x1F4B5), color: '#22C55E' },
    { id: 'hit_every_bonus', label: 'Bonuses', icon: String.fromCodePoint(0x1F381), color: '#F59E0B' },
  ];

  const results = ALL_GOALS.map(goal => {
    const recs = getOptimalCard(cards, category, amount || 100, [goal.id]); // 100 is fine here for rate calculation only
    const best = recs[0];
    if (!best) return null;
    const card = best.card || best;
    return {
      goalId: goal.id,
      label: goal.label,
      icon: goal.icon,
      color: goal.color,
      isActive: userGoals.includes(goal.id),
      cardName: card.nickname || card.issuer,
      lastFour: card.lastFour || '',
      rewardRate: best.rewardRate,
      rewardType: card.rewardType,
      rewardsEarned: best.rewardsEarned,
      cardId: card.id,
    };
  }).filter(Boolean);

  return { results };
}

// ── AI Explanation ──

async function getAiExplanation(data) {
  const stored = await chrome.storage.local.get(['authToken']);
  if (!stored.authToken) return { explanation: '' };

  try {
    const res = await fetch(`${API_BASE}/api/explain-recommendation`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${stored.authToken}`,
      },
      body: JSON.stringify(data),
    });
    if (!res.ok) return { explanation: '' };
    return await res.json();
  } catch {
    return { explanation: '' };
  }
}

// ── Transaction Logging ──

async function logTransaction(transaction) {
  const stored = await chrome.storage.local.get(['authToken']);
  if (!stored.authToken) return { success: false };

  try {
    const res = await fetch(`${API_BASE}/api/transactions`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${stored.authToken}`,
      },
      body: JSON.stringify(transaction),
    });
    return { success: res.ok };
  } catch {
    return { success: false };
  }
}

// ── Retailer Search ──

async function searchRetailers(query) {
  // Use server-synced merchants if available, otherwise use local fallback
  const data = await chrome.storage.local.get(['serverMerchants']);
  const merchants = data.serverMerchants || getLocalMerchantList();

  if (!query || !query.trim()) {
    return { retailers: merchants.filter(m => m.popular).slice(0, 12) };
  }

  const q = query.toLowerCase().trim();
  const results = merchants.filter(m =>
    m.name.toLowerCase().includes(q) ||
    m.domain.includes(q) ||
    m.category.includes(q)
  ).slice(0, 20);

  return { retailers: results };
}

// ── Local Optimizer (ported from lib/optimizer.ts) ──
// This runs entirely in the extension for instant recommendations.
// The full optimizer code is in lib/optimizer.js — imported inline here
// for the service worker context (no ES module imports in MV3 service workers
// without bundling).

function getOptimalCard(cards, category, amount, goals) {
  const activeCards = cards.filter(c => c.isActive !== false);
  if (activeCards.length === 0) return [];

  const recommendations = activeCards.map(card => {
    let bestRate = card.baseRewardRate || 1;
    let reasonParts = [];

    const bonuses = card.categoryBonuses || [];
    const bonus = bonuses.find(b => b.category === category);
    const spendingCapRemaining = bonus?.spendingCap
      ? (bonus.spendingCap - (bonus.currentSpend || 0)) : null;

    if (bonus) {
      const capReached = bonus.spendingCap && (bonus.currentSpend || 0) >= bonus.spendingCap;
      if (!capReached) {
        bestRate = bonus.rate;
        const earned = (amount * bonus.rate) / 100;
        const label = card.rewardType === 'miles'
          ? `${Math.round(earned)} miles`
          : `$${earned.toFixed(2)}`;
        reasonParts.push(
          `${bonus.rate}x ${card.rewardType} on ${category} — earn ${label} on $${amount.toFixed(2)}`
        );
        if (spendingCapRemaining && spendingCapRemaining > 0) {
          reasonParts.push(`$${spendingCapRemaining.toFixed(2)} remaining before cap`);
        }
      } else {
        reasonParts.push(`Cap reached for ${category}, using ${card.baseRewardRate}x base rate`);
      }
    } else {
      const earned = (amount * (card.baseRewardRate || 1)) / 100;
      const label = card.rewardType === 'miles'
        ? `${Math.round(earned)} miles`
        : `$${earned.toFixed(2)}`;
      reasonParts.push(`${card.baseRewardRate || 1}x ${card.rewardType} — earn ${label}`);
    }

    const rewardsEarned = (amount * bestRate) / 100;
    let goalScore = rewardsEarned * 100;

    if (goals && goals.length > 0) {
      if (goals.includes('build_credit')) {
        const limit = card.creditLimit || 0;
        const balance = card.currentBalance || 0;
        if (limit > 0) {
          const newUtil = ((balance + amount) / limit) * 100;
          if (newUtil > 30) {
            goalScore -= (newUtil - 30) * 3;
            reasonParts.push(`utilization would reach ${newUtil.toFixed(0)}% (aim under 30%)`);
          } else {
            goalScore += (30 - newUtil);
            reasonParts.push(`keeps utilization at ${newUtil.toFixed(0)}% — optimal for credit`);
          }
        }
      }

      if (goals.includes('hit_every_bonus')) {
        const hasWelcomeBonus = card.welcomeBonus || card.signupBonus;
        const hasRotatingCategory = (card.categoryBonuses || []).some(b => b.isRotating && b.category === category);
        if (hasWelcomeBonus) {
          goalScore += 50;
          reasonParts.push('working toward welcome bonus — route spend here');
        }
        if (hasRotatingCategory) {
          goalScore += 30;
          reasonParts.push('rotating category active for this purchase');
        }
      }

      if (goals.includes('travel_focus') && card.rewardType === 'miles') {
        goalScore *= 1.5;
        reasonParts.push(`travel priority: earn ${Math.round(rewardsEarned)} miles toward your next trip`);
      }

      if (goals.includes('cashback_focus') && card.rewardType === 'cashback') {
        goalScore *= 1.5;
        reasonParts.push(`cash back priority: $${rewardsEarned.toFixed(2)} straight to your pocket`);
      }

      if (goals.includes('budget_conscious')) {
        const limit = card.creditLimit || 0;
        const balance = card.currentBalance || 0;
        if (limit > 0) {
          const newUtil = ((balance + amount) / limit) * 100;
          if (newUtil > 80) {
            goalScore -= 60;
            reasonParts.push('approaching credit limit — consider a lower-utilization card');
          } else if (newUtil > 50) {
            goalScore -= 20;
            reasonParts.push(`utilization would be ${newUtil.toFixed(0)}% — watch spending on this card`);
          }
        }
      }
    }

    return {
      cardId: card.id,
      card,
      rewardsEarned,
      rewardRate: bestRate,
      reason: reasonParts.join('. ') || `Base ${card.rewardType} rate`,
      isBest: false,
      _goalScore: goalScore,
    };
  });

  recommendations.sort((a, b) => b._goalScore - a._goalScore);
  if (recommendations.length > 0) recommendations[0].isBest = true;
  recommendations.forEach(r => delete r._goalScore);

  return recommendations;
}

// ── Local Merchant Fallback ──
// Used when server sync hasn't happened yet.
// This is a subset — the full list syncs from /api/extension/config.

function getLocalMerchantList() {
  return [
    { name: 'Amazon', domain: 'amazon.com', category: 'online_shopping', icon: '🛒', popular: true },
    { name: 'Walmart', domain: 'walmart.com', category: 'department_store', icon: '🏪', popular: true },
    { name: 'Target', domain: 'target.com', category: 'department_store', icon: '🎯', popular: true },
    { name: 'Best Buy', domain: 'bestbuy.com', category: 'online_shopping', icon: '💻', popular: true },
    { name: 'eBay', domain: 'ebay.com', category: 'online_shopping', icon: '🏷️', popular: true },
    { name: 'Etsy', domain: 'etsy.com', category: 'online_shopping', icon: '🎨', popular: true },
    { name: 'Apple', domain: 'apple.com', category: 'online_shopping', icon: '🍎', popular: true },
    { name: 'Nike', domain: 'nike.com', category: 'online_shopping', icon: '👟', popular: true },
    { name: 'Costco', domain: 'costco.com', category: 'groceries', icon: '📦', popular: true },
    { name: 'Home Depot', domain: 'homedepot.com', category: 'home_improvement', icon: '🔨', popular: true },
    { name: 'Lowes', domain: 'lowes.com', category: 'home_improvement', icon: '🏠', popular: true },
    { name: 'Wayfair', domain: 'wayfair.com', category: 'online_shopping', icon: '🛋️', popular: true },
  ];
}