// lib/amount-extractor.js — Purchase Amount Extraction
// Attempts to find the total purchase amount on checkout pages.
// Returns null if not found (extension defaults to $100 for rate comparison).

const AMOUNT_SELECTORS = [
  // Specific test IDs
  '[data-testid*="total"]',
  '[data-testid*="order-total"]',
  '[data-testid*="grand-total"]',
  '[data-testid*="amount"]',

  // Common class names
  '.order-total', '.cart-total', '.grand-total',
  '.order-summary-total', '.cart-summary-total',
  '.total-price', '.totalPrice', '.order-amount',
  '.checkout-total', '.payment-total',
  '.summary-total-amount', '.cart-total-price',

  // IDs
  '#order-total', '#cart-total', '#grand-total',
  '#total-price', '#totalPrice',

  // Attribute patterns
  '[class*="orderTotal"]', '[class*="cartTotal"]',
  '[class*="grandTotal"]', '[class*="totalPrice"]',
  '[class*="order-total"]', '[class*="cart-total"]',
  '[class*="summary-total"]',
];

const TOTAL_TEXT_PATTERNS = [
  /total/i,
  /order\s*total/i,
  /grand\s*total/i,
  /amount\s*due/i,
  /you\s*pay/i,
  /estimated\s*total/i,
];

const PRICE_REGEX = /\$\s*([\d,]+\.?\d{0,2})/;

function extractAmount(document) {
  // Strategy 1: Direct selector match
  for (const selector of AMOUNT_SELECTORS) {
    try {
      const elements = document.querySelectorAll(selector);
      for (const el of elements) {
        const amount = parsePriceFromElement(el);
        if (amount && amount > 0 && amount < 100000) return amount;
      }
    } catch {}
  }

  // Strategy 2: Find elements containing "Total" text
  const allElements = document.querySelectorAll(
    'td, th, span, div, p, label, dt, dd, li, strong, b'
  );

  for (const el of allElements) {
    const text = el.textContent || '';
    if (text.length > 200) continue; // Skip large text blocks

    for (const pattern of TOTAL_TEXT_PATTERNS) {
      if (pattern.test(text)) {
        // Check this element and its siblings/children for a price
        const amount = parsePriceFromElement(el);
        if (amount && amount > 0 && amount < 100000) return amount;

        // Check next sibling
        const next = el.nextElementSibling;
        if (next) {
          const sibAmount = parsePriceFromElement(next);
          if (sibAmount && sibAmount > 0 && sibAmount < 100000) return sibAmount;
        }

        // Check parent's last child (common in table rows)
        const parent = el.parentElement;
        if (parent) {
          const lastChild = parent.lastElementChild;
          if (lastChild && lastChild !== el) {
            const childAmount = parsePriceFromElement(lastChild);
            if (childAmount && childAmount > 0 && childAmount < 100000) return childAmount;
          }
        }
      }
    }
  }

  // Strategy 3: Look for the largest price on the page (likely the total)
  const allPrices = [];
  for (const el of allElements) {
    const text = (el.textContent || '').trim();
    if (text.length > 50) continue;
    const match = text.match(PRICE_REGEX);
    if (match) {
      const val = parseFloat(match[1].replace(/,/g, ''));
      if (val > 0 && val < 100000) allPrices.push(val);
    }
  }

  if (allPrices.length > 0) {
    // Return the largest price (most likely the order total)
    return Math.max(...allPrices);
  }

  return null;
}

function parsePriceFromElement(el) {
  const text = (el.textContent || '').trim();
  const match = text.match(PRICE_REGEX);
  if (match) {
    return parseFloat(match[1].replace(/,/g, ''));
  }
  return null;
}

if (typeof window !== 'undefined') {
  window.extractAmount = extractAmount;
}