// lib/optimizer.js — Card Optimization Engine
// Ported from SuperPay mobile app's lib/optimizer.ts
// Runs locally in the extension for instant recommendations.
// Exact same scoring logic as the mobile app.

const GOAL_LABELS = {
  maximize_rewards: 'Maximize Rewards',
  build_credit: 'Protect Credit',
  budget_conscious: 'Budget Conscious',
  travel_focus: 'Travel Rewards',
  cashback_focus: 'Cash Back',
  hit_every_bonus: 'Hit Every Bonus',
};

function computeGoalUrgency(goals, cards, amount) {
  const urgency = {};
  const cardsWithLimits = cards.filter(c => c.creditLimit && c.creditLimit > 0);
  const maxUtil = cardsWithLimits.length > 0
    ? Math.max(...cardsWithLimits.map(c => ((c.currentBalance || 0) / (c.creditLimit || 1)) * 100))
    : 0;
  const avgUtil = cardsWithLimits.length > 0
    ? cardsWithLimits.reduce((sum, c) => sum + ((c.currentBalance || 0) / (c.creditLimit || 1)) * 100, 0) / cardsWithLimits.length
    : 0;
  const anyNearLimit = cardsWithLimits.some(c => {
    const util = ((c.currentBalance || 0) + amount) / (c.creditLimit || 1) * 100;
    return util > 60;
  });

  for (const goal of goals) {
    switch (goal) {
      case 'build_credit':
        if (maxUtil > 25) urgency[goal] = 2.0 + (maxUtil - 25) * 0.05;
        else if (maxUtil > 15) urgency[goal] = 1.2;
        else urgency[goal] = 0.8;
        break;
      case 'hit_every_bonus':
        urgency[goal] = 1.8;
        break;
      case 'budget_conscious':
        if (anyNearLimit) urgency[goal] = 2.5;
        else if (maxUtil > 40) urgency[goal] = 1.5;
        else urgency[goal] = 0.8;
        break;
      case 'travel_focus':
        if (avgUtil < 15) urgency[goal] = 1.5;
        else if (avgUtil < 25) urgency[goal] = 1.2;
        else urgency[goal] = 0.7;
        break;
      case 'cashback_focus':
        if (avgUtil < 15) urgency[goal] = 1.5;
        else if (avgUtil < 25) urgency[goal] = 1.2;
        else urgency[goal] = 0.7;
        break;
      case 'maximize_rewards':
        urgency[goal] = 1.0;
        break;
      default:
        urgency[goal] = 1.0;
    }
  }

  return urgency;
}

function getOptimalCard(cards, category, amount, goals) {
  const activeCards = cards.filter(c => c.isActive !== false);
  if (activeCards.length === 0) return [];

  const activeGoals = goals && goals.length > 0 ? goals : [];
  const goalUrgency = activeGoals.length > 0
    ? computeGoalUrgency(activeGoals, activeCards, amount)
    : {};

  const recommendations = activeCards.map(card => {
    let bestRate = card.baseRewardRate || 1;
    let reasonParts = [];
    const goalImpacts = {};

    const bonuses = card.categoryBonuses || [];
    const bonus = bonuses.find(b => b.category === category);
    const spendingCapRemaining = bonus?.spendingCap
      ? (bonus.spendingCap - (bonus.currentSpend || 0))
      : null;

    if (bonus) {
      const capReached = bonus.spendingCap && (bonus.currentSpend || 0) >= bonus.spendingCap;
      if (!capReached) {
        bestRate = bonus.rate;
        const earned = (amount * bonus.rate) / 100;
        const label = card.rewardType === 'miles'
          ? `${Math.round(earned)} miles`
          : card.rewardType === 'points'
            ? `${Math.round(earned)} points`
            : `$${earned.toFixed(2)}`;
        reasonParts.push(
          `${bonus.rate}x ${card.rewardType} on ${category} — earn ${label} on this $${amount.toFixed(2)} purchase`
        );
        if (spendingCapRemaining && spendingCapRemaining > 0) {
          reasonParts.push(`$${spendingCapRemaining.toFixed(2)} remaining before quarterly cap`);
        }
      } else {
        reasonParts.push(`Cap reached for ${category}, using ${card.baseRewardRate}x base rate`);
      }
    } else {
      const earned = (amount * (card.baseRewardRate || 1)) / 100;
      const label = card.rewardType === 'miles'
        ? `${Math.round(earned)} miles`
        : card.rewardType === 'points'
          ? `${Math.round(earned)} points`
          : `$${earned.toFixed(2)}`;
      reasonParts.push(
        `${card.baseRewardRate || 1}x ${card.rewardType} — earn ${label} on this $${amount.toFixed(2)} purchase`
      );
    }

    const rewardsEarned = (amount * bestRate) / 100;
    let goalScore = rewardsEarned * 100;

    if (activeGoals.length > 0) {
      if (activeGoals.includes('build_credit')) {
        const limit = card.creditLimit || 0;
        const balance = card.currentBalance || 0;
        if (limit > 0) {
          const newUtil = ((balance + amount) / limit) * 100;
          const weight = goalUrgency['build_credit'] || 1;

          if (newUtil > 30) {
            const penalty = (newUtil - 30) * 3 * weight;
            goalScore -= penalty;
            goalImpacts['build_credit'] = -penalty;
            reasonParts.push(`utilization would reach ${newUtil.toFixed(0)}% (aim under 30%)`);
          } else {
            const bonusScore = (30 - newUtil) * weight;
            goalScore += bonusScore;
            goalImpacts['build_credit'] = bonusScore;
            reasonParts.push(`keeps utilization at ${newUtil.toFixed(0)}% — optimal for credit`);
          }
        }
      }

      if (activeGoals.includes('hit_every_bonus')) {
        const weight = goalUrgency['hit_every_bonus'] || 1;
        const hasWelcomeBonus = card.welcomeBonus || card.signupBonus;
        const hasRotatingCategory = (card.categoryBonuses || []).some(b => b.isRotating && b.category === category);
        if (hasWelcomeBonus) {
          const boost = 50 * weight;
          goalScore += boost;
          goalImpacts['hit_every_bonus'] = (goalImpacts['hit_every_bonus'] || 0) + boost;
          reasonParts.push('working toward welcome bonus — route spend here');
        }
        if (hasRotatingCategory) {
          const boost = 30 * weight;
          goalScore += boost;
          goalImpacts['hit_every_bonus'] = (goalImpacts['hit_every_bonus'] || 0) + boost;
          reasonParts.push('rotating category active for this purchase');
        }
      }

      if (activeGoals.includes('travel_focus')) {
        const weight = goalUrgency['travel_focus'] || 1;
        if (card.rewardType === 'miles') {
          const boost = goalScore * 0.5 * weight;
          goalScore += boost;
          goalImpacts['travel_focus'] = boost;
          reasonParts.push(`travel priority: earn ${Math.round(rewardsEarned)} miles toward your next trip`);
        }
      }

      if (activeGoals.includes('cashback_focus')) {
        const weight = goalUrgency['cashback_focus'] || 1;
        if (card.rewardType === 'cashback') {
          const boost = goalScore * 0.5 * weight;
          goalScore += boost;
          goalImpacts['cashback_focus'] = boost;
          reasonParts.push(`cash back priority: $${rewardsEarned.toFixed(2)} straight to your pocket`);
        }
      }

      if (activeGoals.includes('budget_conscious')) {
        const limit = card.creditLimit || 0;
        const balance = card.currentBalance || 0;
        const weight = goalUrgency['budget_conscious'] || 1;
        if (limit > 0) {
          const newUtil = ((balance + amount) / limit) * 100;
          if (newUtil > 80) {
            const penalty = 60 * weight;
            goalScore -= penalty;
            goalImpacts['budget_conscious'] = -penalty;
            reasonParts.push('approaching credit limit — consider a lower-utilization card');
          } else if (newUtil > 50) {
            const penalty = 20 * weight;
            goalScore -= penalty;
            goalImpacts['budget_conscious'] = -penalty;
            reasonParts.push(`utilization would be ${newUtil.toFixed(0)}% — watch your spending on this card`);
          }
        }
      }
    }

    let primaryGoal;
    if (Object.keys(goalImpacts).length > 0) {
      const sorted = Object.entries(goalImpacts).sort((a, b) => Math.abs(b[1]) - Math.abs(a[1]));
      primaryGoal = sorted[0][0];
    }

    return {
      cardId: card.id,
      card,
      rewardsEarned,
      rewardRate: bestRate,
      reason: reasonParts.join('. ') || `Base ${card.rewardType} rate`,
      isBest: false,
      primaryGoal,
      activeGoals: activeGoals.length > 0 ? [...activeGoals] : undefined,
      _goalScore: goalScore,
    };
  });

  recommendations.sort((a, b) => b._goalScore - a._goalScore);
  if (recommendations.length > 0) recommendations[0].isBest = true;
  recommendations.forEach(r => delete r._goalScore);

  return recommendations;
}

if (typeof window !== 'undefined') {
  window.getOptimalCard = getOptimalCard;
  window.GOAL_LABELS = GOAL_LABELS;
}
