// popup.js — SuperPay Extension Popup Logic

document.addEventListener('DOMContentLoaded', async () => {
  const loginView = document.getElementById('login-view');
  const mainView = document.getElementById('main-view');
  const loginForm = document.getElementById('login-form');
  const loginError = document.getElementById('login-error');
  const loginBtn = document.getElementById('login-btn');
  const logoutBtn = document.getElementById('logout-btn');
  const syncBtn = document.getElementById('sync-btn');
  const settingsBtn = document.getElementById('settings-btn');
  const cardsList = document.getElementById('cards-list');
  const searchInput = document.getElementById('search-input');
  const searchResults = document.getElementById('search-results');
  const currentRec = document.getElementById('current-rec');
  const currentRecContent = document.getElementById('current-rec-content');
  const userNameEl = document.getElementById('user-name');
  const userEmailEl = document.getElementById('user-email');
  const goalsPills = document.getElementById('goals-pills');

  function svgIcon(paths, color) {
    return `<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="${color}" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">${paths}</svg>`;
  }

  const GOAL_DEFS = [
    { id: 'maximize_rewards', label: 'Max Rewards', icon: svgIcon('<polygon points="12 2 15.09 8.26 22 9.27 17 14.14 18.18 21.02 12 17.77 5.82 21.02 7 14.14 2 9.27 8.91 8.26 12 2"/>', '#FFB800'), color: '#FFB800' },
    { id: 'build_credit', label: 'Protect Credit', icon: svgIcon('<polyline points="23 6 13.5 15.5 8.5 10.5 1 18"/><polyline points="17 6 23 6 23 12"/>', '#00C896'), color: '#00C896' },
    { id: 'budget_conscious', label: 'Budget', icon: svgIcon('<path d="M12 1a3 3 0 0 0-3 3v8a3 3 0 0 0 6 0V4a3 3 0 0 0-3-3z"/><path d="M19 10v2a7 7 0 0 1-14 0v-2"/><line x1="12" y1="19" x2="12" y2="23"/>', '#FF6B6B'), color: '#FF6B6B' },
    { id: 'travel_focus', label: 'Travel', icon: svgIcon('<path d="M17.8 19.2L16 11l3.5-3.5C21 6 21.5 4 21 3s-3-1-4.5.5L13 7 4.8 5.2c-.5-.1-1 .1-1.3.5l-.3.5 5.1 3 1 1-4.2.8-.8-.8-1.5 1.5 3 2 2 3 1.5-1.5-.8-.8.8-4.2 1 1 3 5.1.5-.3c.4-.3.6-.8.5-1.3z"/>', '#A855F7'), color: '#A855F7' },
    { id: 'cashback_focus', label: 'Cash Back', icon: svgIcon('<line x1="12" y1="1" x2="12" y2="23"/><path d="M17 5H9.5a3.5 3.5 0 0 0 0 7h5a3.5 3.5 0 0 1 0 7H6"/>', '#22C55E'), color: '#22C55E' },
    { id: 'hit_every_bonus', label: 'Bonuses', icon: svgIcon('<polyline points="20 12 20 22 4 22 4 12"/><rect x="2" y="7" width="20" height="5"/><line x1="12" y1="22" x2="12" y2="7"/><path d="M12 7H7.5a2.5 2.5 0 0 1 0-5C11 2 12 7 12 7z"/><path d="M12 7h4.5a2.5 2.5 0 0 0 0-5C13 2 12 7 12 7z"/>', '#F59E0B'), color: '#F59E0B' },
  ];

  function getCategoryIcon(category) {
    const icons = {
      dining: '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="#8B9DC3" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M18 8h1a4 4 0 0 1 0 8h-1"/><path d="M2 8h16v9a4 4 0 0 1-4 4H6a4 4 0 0 1-4-4V8z"/><line x1="6" y1="1" x2="6" y2="4"/><line x1="10" y1="1" x2="10" y2="4"/><line x1="14" y1="1" x2="14" y2="4"/></svg>',
      travel: '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="#8B9DC3" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M17.8 19.2L16 11l3.5-3.5C21 6 21.5 4 21 3s-3-1-4.5.5L13 7 4.8 5.2c-.5-.1-1 .1-1.3.5l-.3.5 5.1 3 1 1-4.2.8-.8-.8-1.5 1.5 3 2 2 3 1.5-1.5-.8-.8.8-4.2 1 1 3 5.1.5-.3c.4-.3.6-.8.5-1.3z"/></svg>',
      groceries: '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="#8B9DC3" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="9" cy="21" r="1"/><circle cx="20" cy="21" r="1"/><path d="M1 1h4l2.68 13.39a2 2 0 0 0 2 1.61h9.72a2 2 0 0 0 2-1.61L23 6H6"/></svg>',
      gas: '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="#8B9DC3" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M3 22V5a2 2 0 0 1 2-2h8a2 2 0 0 1 2 2v17"/><path d="M15 10h2a2 2 0 0 1 2 2v2a2 2 0 0 0 2 2v0a2 2 0 0 0 2-2V9l-3-3"/><rect x="5" y="7" width="8" height="5"/><line x1="1" y1="22" x2="23" y2="22"/></svg>',
      entertainment: '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="#8B9DC3" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><rect x="2" y="2" width="20" height="20" rx="2.18" ry="2.18"/><line x1="7" y1="2" x2="7" y2="22"/><line x1="17" y1="2" x2="17" y2="22"/><line x1="2" y1="12" x2="22" y2="12"/><line x1="2" y1="7" x2="7" y2="7"/><line x1="2" y1="17" x2="7" y2="17"/><line x1="17" y1="7" x2="22" y2="7"/><line x1="17" y1="17" x2="22" y2="17"/></svg>',
      streaming: '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="#8B9DC3" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polygon points="5 3 19 12 5 21 5 3"/></svg>',
      shopping: '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="#8B9DC3" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M6 2L3 6v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2V6l-3-4z"/><line x1="3" y1="6" x2="21" y2="6"/><path d="M16 10a4 4 0 0 1-8 0"/></svg>',
      online_shopping: '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="#8B9DC3" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M6 2L3 6v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2V6l-3-4z"/><line x1="3" y1="6" x2="21" y2="6"/><path d="M16 10a4 4 0 0 1-8 0"/></svg>',
      electronics: '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="#8B9DC3" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><rect x="2" y="3" width="20" height="14" rx="2" ry="2"/><line x1="8" y1="21" x2="16" y2="21"/><line x1="12" y1="17" x2="12" y2="21"/></svg>',
      home_improvement: '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="#8B9DC3" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M3 9l9-7 9 7v11a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2z"/><polyline points="9 22 9 12 15 12 15 22"/></svg>',
      utilities: '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="#8B9DC3" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polygon points="13 2 3 14 12 14 11 22 21 10 12 10 13 2"/></svg>',
      health: '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="#8B9DC3" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M22 12h-4l-3 9L9 3l-3 9H2"/></svg>',
      fitness: '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="#8B9DC3" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M22 12h-4l-3 9L9 3l-3 9H2"/></svg>',
      insurance: '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="#8B9DC3" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M12 22s8-4 8-10V5l-8-3-8 3v7c0 6 8 10 8 10z"/></svg>',
    };
    return icons[category] || icons.shopping;
  }

  let activeGoals = [];

  const VALUE_CPP = { miles: 1.5, points: 1.2, cashback: 1.0 };

  function getEstimatedValue(rate, rewardType) {
    const cpp = VALUE_CPP[rewardType] || 1.0;
    const effectivePct = rate * cpp;
    return effectivePct;
  }

  function formatValueLabel(rate, rewardType, amount) {
    const cpp = VALUE_CPP[rewardType] || 1.0;
    const effectivePct = rate * cpp;
    if (amount && amount > 0) {
      const dollarValue = (amount * rate * cpp) / 100;
      return `\u2248 $${dollarValue.toFixed(2)} value`;
    }
    return `\u2248 ${effectivePct.toFixed(1)}\u00A2/$ value`;
  }

  // Check auth state
  const authState = await chrome.runtime.sendMessage({ type: 'GET_AUTH_STATE' });

  if (authState.loggedIn) {
    showMainView(authState.user);
  } else {
    showLoginView();
  }

  // ── Login ──

  loginForm.addEventListener('submit', async (e) => {
    e.preventDefault();
    loginError.style.display = 'none';
    loginBtn.disabled = true;
    loginBtn.textContent = 'Signing in...';

    const email = document.getElementById('email').value.trim();
    const password = document.getElementById('password').value;

    const result = await chrome.runtime.sendMessage({
      type: 'LOGIN',
      email,
      password,
    });

    if (result.success) {
      showMainView(result.user);
    } else {
      loginError.textContent = result.error || 'Login failed';
      loginError.style.display = 'block';
      loginBtn.disabled = false;
      loginBtn.textContent = 'Sign In';
    }
  });

  // ── Logout ──

  logoutBtn.addEventListener('click', async () => {
    await chrome.runtime.sendMessage({ type: 'LOGOUT' });
    showLoginView();
  });

  // ── Sync ──

  const syncSvgDefault = '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M21.5 2v6h-6M2.5 22v-6h6"/><path d="M2.5 11.5a10 10 0 0 1 18.4-4.5M21.5 12.5a10 10 0 0 1-18.4 4.5"/></svg>';
  const syncSvgLoading = '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="spin-icon"><path d="M21 12a9 9 0 1 1-6.219-8.56"/></svg>';

  syncBtn.addEventListener('click', async () => {
    syncBtn.innerHTML = syncSvgLoading;
    await chrome.runtime.sendMessage({ type: 'SYNC_CARDS' });
    await loadCards();
    syncBtn.innerHTML = syncSvgDefault;
  });

  // ── Settings ──

  settingsBtn.addEventListener('click', () => {
    chrome.runtime.openOptionsPage();
  });

  // ── Search ──

  let searchTimeout;
  searchInput.addEventListener('input', () => {
    clearTimeout(searchTimeout);
    const query = searchInput.value.trim();

    if (!query) {
      searchResults.style.display = 'none';
      return;
    }

    searchTimeout = setTimeout(async () => {
      // Check if it looks like a URL
      if (query.includes('.') && !query.includes(' ')) {
        const result = await chrome.runtime.sendMessage({
          type: 'PARSE_URL',
          url: query,
        });

        if (result.found) {
          searchResults.innerHTML = `
            <div class="search-result-item search-result-match">
              <div class="sr-name">${result.merchant}</div>
              <div class="sr-category">${result.category}</div>
              ${result.bestCard ? `
                <div class="sr-card">
                  Use: ${result.bestCard.nickname} (${result.bestCard.rewardRate}x ${result.bestCard.rewardType})
                </div>
              ` : ''}
            </div>
          `;
          searchResults.style.display = 'block';
          return;
        }
      }

      // Regular retailer search
      const result = await chrome.runtime.sendMessage({
        type: 'SEARCH_RETAILERS',
        query,
      });

      if (result.retailers && result.retailers.length > 0) {
        searchResults.innerHTML = result.retailers.map(r => `
          <div class="search-result-item" data-domain="${r.domain}" data-category="${r.category}" data-name="${r.name}">
            <span class="sr-icon">${getCategoryIcon(r.category)}</span>
            <div class="sr-info">
              <div class="sr-name">${r.name}</div>
              <div class="sr-category">${r.category.replace(/_/g, ' ')}</div>
            </div>
          </div>
        `).join('');

        // Click handler for search results
        searchResults.querySelectorAll('.search-result-item[data-domain]').forEach(item => {
          item.addEventListener('click', async () => {
            const name = item.dataset.name;
            const category = item.dataset.category;
            const rec = await chrome.runtime.sendMessage({
              type: 'GET_RECOMMENDATION',
              merchant: name,
              category: category,
              amount: 100,
            });

            if (rec.recommendation) {
              const card = rec.recommendation.card;
              const rewardLabel = card.rewardType === 'miles'
                ? `${Math.round(rec.recommendation.rewardsEarned)} miles`
                : `$${rec.recommendation.rewardsEarned.toFixed(2)}`;

              const existingCategory = item.dataset.category || 'shopping';
              item.innerHTML = `
                <span class="sr-icon">${getCategoryIcon(existingCategory)}</span>
                <div class="sr-info">
                  <div class="sr-name">${name}</div>
                  <div class="sr-best-card">
                    Best: ${card.nickname || card.issuer}
                    (${rec.recommendation.rewardRate}x · ${rewardLabel})
                  </div>
                </div>
              `;
              item.classList.add('search-result-match');
            }
          });
        });

        searchResults.style.display = 'block';
      } else {
        searchResults.innerHTML = '<div class="search-empty">No retailers found</div>';
        searchResults.style.display = 'block';
      }
    }, 300);
  });

  // ── Views ──

  function showLoginView() {
    loginView.style.display = 'block';
    mainView.style.display = 'none';
  }

  async function showMainView(user) {
    loginView.style.display = 'none';
    mainView.style.display = 'block';

    if (user) {
      userNameEl.textContent = user.name || '';
      userEmailEl.textContent = user.email || '';
    }

    await loadGoals();
    await loadCards();
    await loadRoadmap();
    await checkCurrentPage();
  }

  async function loadRoadmap() {
    const roadmapSection = document.getElementById('roadmap-section');
    const roadmapContent = document.getElementById('roadmap-content');
    if (!roadmapSection || !roadmapContent) return;

    try {
      const authData = await chrome.storage.local.get(['subscriptionTier']);
      const tier = authData.subscriptionTier || '';
      const isProPlus = tier === 'pro_plus' || tier === 'pro+';

      if (isProPlus) {
        try {
          const response = await chrome.runtime.sendMessage({ type: 'GET_ROADMAPS' });
          const roadmaps = response?.roadmaps || [];
          const activeRoadmap = roadmaps.find(r => r.status === 'active') || roadmaps[0];

          if (activeRoadmap) {
            const goalName = activeRoadmap.goalDescription || activeRoadmap.goalType || 'Rewards Goal';
            const goalType = (activeRoadmap.goalType || '').replace(/_/g, ' ');
            const target = activeRoadmap.targetValue || 0;
            const current = activeRoadmap.currentProgress || 0;
            const pct = target > 0 ? Math.min(Math.round((current / target) * 100), 100) : 0;
            const remaining = Math.max(target - current, 0);
            const isPoints = activeRoadmap.goalType === 'points_milestone';
            const remainingLabel = isPoints
              ? `${remaining.toLocaleString()} pts remaining`
              : `$${remaining.toLocaleString()} remaining`;

            roadmapContent.innerHTML = `
              <div class="roadmap-progress-card">
                <div class="roadmap-pro-badge">PRO+ ROADMAP</div>
                <div class="roadmap-goal-name">${goalName}</div>
                <div class="roadmap-goal-type">${goalType}</div>
                <div class="roadmap-progress-bar-wrap">
                  <div class="roadmap-progress-bar-fill" style="width: ${pct}%"></div>
                </div>
                <div class="roadmap-progress-stats">
                  <span class="roadmap-progress-pct">${pct}% complete</span>
                  <span class="roadmap-progress-remaining">${remainingLabel}</span>
                </div>
              </div>
            `;
            roadmapSection.style.display = 'block';
          } else {
            roadmapSection.style.display = 'none';
          }
        } catch {
          roadmapSection.style.display = 'none';
        }
      } else {
        roadmapContent.innerHTML = `
          <div class="roadmap-upsell-card">
            <div class="roadmap-upsell-icon">
              <svg viewBox="0 0 24 24" fill="none" stroke="#FFB800" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                <polygon points="12 2 15.09 8.26 22 9.27 17 14.14 18.18 21.02 12 17.77 5.82 21.02 7 14.14 2 9.27 8.91 8.26 12 2"/>
              </svg>
            </div>
            <div class="roadmap-upsell-title">Unlock AI Rewards Roadmap</div>
            <div class="roadmap-upsell-desc">Turn any goal into a step-by-step plan. Get smart card recommendations, monthly action plans, and progress tracking powered by AI.</div>
            <a class="roadmap-upsell-link" id="roadmap-learn-more">Learn More
              <svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round"><polyline points="9 18 15 12 9 6"/></svg>
            </a>
          </div>
        `;
        roadmapSection.style.display = 'block';

        document.getElementById('roadmap-learn-more')?.addEventListener('click', (e) => {
          e.preventDefault();
          chrome.tabs.create({ url: 'https://superpayrewards.com' });
        });
      }
    } catch {
      roadmapSection.style.display = 'none';
    }
  }

  async function loadGoals() {
    const data = await chrome.storage.local.get(['goals']);
    activeGoals = data.goals || [];
    renderGoals();
  }

  function renderGoals() {
    goalsPills.innerHTML = GOAL_DEFS.map(g => {
      const isActive = activeGoals.includes(g.id);
      return `<button class="goal-pill ${isActive ? 'goal-pill-active' : ''}" data-goal="${g.id}" style="${isActive ? `background:${g.color}20;border-color:${g.color};color:${g.color}` : ''}"><span class="goal-pill-icon">${g.icon}</span><span class="goal-pill-label">${g.label}</span></button>`;
    }).join('');

    goalsPills.querySelectorAll('.goal-pill').forEach(pill => {
      pill.addEventListener('click', () => toggleGoal(pill.dataset.goal));
    });
  }

  async function toggleGoal(goalId) {
    const idx = activeGoals.indexOf(goalId);
    if (idx >= 0) {
      activeGoals.splice(idx, 1);
    } else {
      activeGoals.push(goalId);
    }

    renderGoals();

    await chrome.runtime.sendMessage({ type: 'UPDATE_GOALS', goals: activeGoals });

    await checkCurrentPage();
  }

  async function loadCards() {
    const result = await chrome.runtime.sendMessage({ type: 'GET_CARDS' });
    const cards = result.cards || [];

    if (cards.length === 0) {
      cardsList.innerHTML = `
        <div class="empty-state">
          No cards yet. Add cards in the SuperPay app to get started.
        </div>
      `;
      return;
    }

    cardsList.innerHTML = cards
      .filter(c => c.isActive)
      .slice(0, 5)
      .map(card => {
        const lastFour = card.lastFour ? `····${card.lastFour}` : '';
        const valLabel = formatValueLabel(card.baseRewardRate, card.rewardType);
        const valHtml = valLabel ? `<span class="card-item-value">${valLabel}</span>` : '';
        return `
          <div class="card-item">
            <div class="card-item-left">
              <div class="card-item-name">${card.nickname || card.issuer}</div>
              <div class="card-item-details">${lastFour} · ${card.baseRewardRate}x ${card.rewardType} ${valHtml}</div>
            </div>
            <div class="card-item-rate">${card.baseRewardRate}x</div>
          </div>
        `;
      })
      .join('');

    if (cards.filter(c => c.isActive).length > 5) {
      cardsList.innerHTML += `<div class="cards-more">+${cards.filter(c => c.isActive).length - 5} more cards in app</div>`;
    }
  }

  async function checkCurrentPage() {
    try {
      const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
      if (!tab?.url) return;

      let pageAmount = null;
      try {
        const amountResponse = await chrome.tabs.sendMessage(tab.id, { type: 'GET_PAGE_AMOUNT' });
        if (amountResponse?.amount && amountResponse.amount > 0) {
          pageAmount = amountResponse.amount;
        }
      } catch {}

      const result = await chrome.runtime.sendMessage({
        type: 'PARSE_URL',
        url: tab.url,
      });

      if (result.found && result.bestCard) {
        currentRec.style.display = 'block';
        const rate = result.bestCard.rewardRate;
        const rewardType = result.bestCard.rewardType;
        let rewardLabel = rewardType === 'miles'
          ? `${rate}x miles`
          : `${rate}x cash back`;

        let earningsHtml = '';
        if (pageAmount && pageAmount > 0) {
          const earned = (pageAmount * rate) / 100;
          const earnedLabel = rewardType === 'miles'
            ? `~${Math.round(earned)} miles`
            : rewardType === 'points'
              ? `~${Math.round(earned)} pts`
              : `~$${earned.toFixed(2)} back`;
          earningsHtml = `<span class="rec-earnings"> · ${earnedLabel} on $${pageAmount.toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2})}</span>`;
        }

        const valueContext = formatValueLabel(rate, rewardType, pageAmount);
        const valueHtml = valueContext ? `<span class="rec-value-context">${valueContext}</span>` : '';

        currentRecContent.innerHTML = `
          <div class="rec-merchant">${result.merchant}</div>
          <div class="rec-category">${(result.category || '').replace(/_/g, ' ')}</div>
          <div class="rec-best">
            <span class="rec-best-label">BEST CARD:</span>
            <span class="rec-best-name">${result.bestCard.nickname}</span>
            <span class="rec-best-rate">${rewardLabel}</span>
            ${valueHtml}
            ${earningsHtml}
          </div>
        `;

        result._pageAmount = pageAmount;
        fetchCurrentPageAiExplanation(result);
        fetchMultiGoalRecommendations(result.category, pageAmount);
      }
    } catch {}
  }

  async function fetchMultiGoalRecommendations(category, pageAmount) {
    const section = document.getElementById('multi-goal-section');
    const list = document.getElementById('multi-goal-list');
    const toggle = document.getElementById('multi-goal-toggle');
    const arrow = document.getElementById('multi-goal-arrow');
    if (!section || !list || !toggle) return;

    const response = await chrome.runtime.sendMessage({
      type: 'GET_MULTI_GOAL_RECOMMENDATION',
      category,
      amount: pageAmount || null,
    });

    if (!response?.results || response.results.length === 0) return;

    section.style.display = 'block';

    list.innerHTML = response.results.map(r => {
      const rewardLabel = r.rewardType === 'miles'
        ? `${r.rewardRate}x miles`
        : r.rewardType === 'points'
          ? `${r.rewardRate}x points`
          : `${r.rewardRate}x cashback`;
      const valLabel = formatValueLabel(r.rewardRate, r.rewardType, pageAmount);
      const valHtml = valLabel ? `<span class="mg-card-value">${valLabel}</span>` : '';
      const activeClass = r.isActive ? 'mg-row-active' : '';
      const activeDot = r.isActive ? '<span class="mg-active-dot"></span>' : '';
      const goalDef = GOAL_DEFS.find(g => g.id === r.goalId);
      const goalIcon = goalDef ? goalDef.icon : svgIcon('<circle cx="12" cy="12" r="10"/>', r.color);
      return `<div class="mg-row ${activeClass}">
        <div class="mg-goal" style="color:${r.color}">
          <span class="mg-goal-icon">${goalIcon}</span>
          <span class="mg-goal-label">${r.label}</span>
          ${activeDot}
        </div>
        <div class="mg-card">
          <span class="mg-card-name">${r.cardName}</span>
          <span class="mg-card-rate">${rewardLabel}</span>
          ${valHtml}
        </div>
      </div>`;
    }).join('');

    let expanded = false;
    const newToggle = toggle.cloneNode(true);
    toggle.parentNode.replaceChild(newToggle, toggle);
    newToggle.addEventListener('click', () => {
      expanded = !expanded;
      list.style.display = expanded ? 'block' : 'none';
      const arrowEl = document.getElementById('multi-goal-arrow');
      if (arrowEl) arrowEl.textContent = expanded ? '\u25B2' : '\u25BC';
    });
  }

  async function fetchCurrentPageAiExplanation(result) {
    const aiSection = document.getElementById('current-rec-ai');
    const aiText = document.getElementById('current-rec-ai-text');
    if (!aiSection || !aiText) return;

    aiSection.style.display = 'block';
    aiText.innerHTML = '<div class="rec-ai-shimmer"><div class="rec-ai-shimmer-line"></div><div class="rec-ai-shimmer-line rec-ai-shimmer-short"></div></div>';

    try {
      const stored = await chrome.storage.local.get(['goals']);
      const userGoals = stored.goals || [];

      const pageAmount = result._pageAmount || null;

      const response = await chrome.runtime.sendMessage({
        type: 'GET_AI_EXPLANATION',
        data: {
          bestCard: {
            nickname: result.bestCard.nickname,
            rewardRate: result.bestCard.rewardRate,
            rewardType: result.bestCard.rewardType,
            rewardsEarned: result.bestCard.rewardsEarned || 0,
          },
          otherCards: [],
          merchant: result.merchant,
          category: result.category,
          amount: pageAmount,
          goals: userGoals,
        },
      });

      if (response?.explanation) {
        aiText.textContent = response.explanation;
      } else {
        aiSection.style.display = 'none';
      }
    } catch {
      aiSection.style.display = 'none';
    }
  }

  // Open signup link
  document.getElementById('open-signup')?.addEventListener('click', (e) => {
    e.preventDefault();
    chrome.tabs.create({ url: 'https://superpayrewards.com' });
  });
});