// content.js — SuperPay Extension Content Script
// Injected on all pages. Detects checkouts and renders the recommendation overlay.

(function() {
  'use strict';

  // Prevent double-injection
  if (window.__superpay_injected) return;
  window.__superpay_injected = true;

  let overlayElement = null;
  let overlayTimeout = null;
  let currentRecommendation = null;
  let cachedServerMerchants = null;

  const VALUE_CPP = { miles: 1.5, points: 1.2, cashback: 1.0 };

  function svgIcon(paths, color, size) {
    const s = size || 14;
    return `<svg width="${s}" height="${s}" viewBox="0 0 24 24" fill="none" stroke="${color}" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">${paths}</svg>`;
  }

  const ICON_SETTINGS = '<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="3"/><path d="M19.4 15a1.65 1.65 0 0 0 .33 1.82l.06.06a2 2 0 1 1-2.83 2.83l-.06-.06a1.65 1.65 0 0 0-1.82-.33 1.65 1.65 0 0 0-1 1.51V21a2 2 0 0 1-4 0v-.09A1.65 1.65 0 0 0 9 19.4a1.65 1.65 0 0 0-1.82.33l-.06.06a2 2 0 1 1-2.83-2.83l.06-.06A1.65 1.65 0 0 0 4.68 15a1.65 1.65 0 0 0-1.51-1H3a2 2 0 0 1 0-4h.09A1.65 1.65 0 0 0 4.6 9a1.65 1.65 0 0 0-.33-1.82l-.06-.06a2 2 0 1 1 2.83-2.83l.06.06A1.65 1.65 0 0 0 9 4.68a1.65 1.65 0 0 0 1-1.51V3a2 2 0 0 1 4 0v.09a1.65 1.65 0 0 0 1 1.51 1.65 1.65 0 0 0 1.82-.33l.06-.06a2 2 0 1 1 2.83 2.83l-.06.06A1.65 1.65 0 0 0 19.4 9a1.65 1.65 0 0 0 1.51 1H21a2 2 0 0 1 0 4h-.09a1.65 1.65 0 0 0-1.51 1z"/></svg>';
  const ICON_AI = '<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="#FFB800" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M12 2L9 9l-7 1 5 5-1.5 7L12 18l6.5 4L17 15l5-5-7-1z"/></svg>';
  const ICON_CHECK = '<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round"><polyline points="20 6 9 17 4 12"/></svg>';

  const GOAL_SVG_MAP = {
    'maximize_rewards': (c) => svgIcon('<polygon points="12 2 15.09 8.26 22 9.27 17 14.14 18.18 21.02 12 17.77 5.82 21.02 7 14.14 2 9.27 8.91 8.26 12 2"/>', c),
    'build_credit': (c) => svgIcon('<polyline points="23 6 13.5 15.5 8.5 10.5 1 18"/><polyline points="17 6 23 6 23 12"/>', c),
    'budget_conscious': (c) => svgIcon('<path d="M12 1a3 3 0 0 0-3 3v8a3 3 0 0 0 6 0V4a3 3 0 0 0-3-3z"/><path d="M19 10v2a7 7 0 0 1-14 0v-2"/><line x1="12" y1="19" x2="12" y2="23"/>', c),
    'travel_focus': (c) => svgIcon('<path d="M17.8 19.2L16 11l3.5-3.5C21 6 21.5 4 21 3s-3-1-4.5.5L13 7 4.8 5.2c-.5-.1-1 .1-1.3.5l-.3.5 5.1 3 1 1-4.2.8-.8-.8-1.5 1.5 3 2 2 3 1.5-1.5-.8-.8.8-4.2 1 1 3 5.1.5-.3c.4-.3.6-.8.5-1.3z"/>', c),
    'cashback_focus': (c) => svgIcon('<line x1="12" y1="1" x2="12" y2="23"/><path d="M17 5H9.5a3.5 3.5 0 0 0 0 7h5a3.5 3.5 0 0 1 0 7H6"/>', c),
    'hit_every_bonus': (c) => svgIcon('<polyline points="20 12 20 22 4 22 4 12"/><rect x="2" y="7" width="20" height="5"/><line x1="12" y1="22" x2="12" y2="7"/><path d="M12 7H7.5a2.5 2.5 0 0 1 0-5C11 2 12 7 12 7z"/><path d="M12 7h4.5a2.5 2.5 0 0 0 0-5C13 2 12 7 12 7z"/>', c),
  };

  function escapeHtml(str) {
    return String(str)
      .replace(/&/g, '&amp;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;')
      .replace(/"/g, '&quot;')
      .replace(/'/g, '&#39;');
  }

  function formatValueLabel(rate, rewardType, amount) {
    const cpp = VALUE_CPP[rewardType] || 1.0;
    const effectivePct = rate * cpp;
    if (amount && amount > 0) {
      const dollarValue = (amount * rate * cpp) / 100;
      return `\u2248 $${dollarValue.toFixed(2)} value`;
    }
    return `\u2248 ${effectivePct.toFixed(1)}\u00A2/$ value`;
  }

  try {
    chrome.storage.local.get(['serverMerchants'], (data) => {
      if (data.serverMerchants && Array.isArray(data.serverMerchants)) {
        cachedServerMerchants = {};
        for (const m of data.serverMerchants) {
          if (m.domain) {
            const d = m.domain.replace(/^www\./, '').toLowerCase();
            cachedServerMerchants[d] = { name: m.name, category: m.category };
          }
        }
      }
    });
  } catch {}

  // ── Checkout Detection Constants (inlined for content script) ──

  const CHECKOUT_URL_PATTERNS = [
    /\/checkout/i, /\/cart/i, /\/basket/i, /\/order/i, /\/payment/i,
    /\/pay\b/i, /\/billing/i, /\/purchase/i, /\/subscribe/i,
    /\/gp\/buy/i, /\/chkout/i, /\/secure/i,
    /\/place-order/i, /\/confirm-order/i,
  ];

  const PAYMENT_BUTTON_TEXTS = [
    'place order', 'place your order', 'complete purchase', 'complete order',
    'pay now', 'pay with', 'submit order', 'confirm order', 'confirm purchase',
    'buy now', 'proceed to checkout', 'continue to payment', 'review order',
    'checkout', 'purchase', 'subscribe',
  ];

  const CARD_INPUT_SELECTORS = [
    'input[autocomplete="cc-number"]', 'input[autocomplete="cc-name"]',
    'input[name*="card"]', 'input[name*="credit"]', 'input[name*="cc-"]',
    'input[id*="card-number"]', 'input[id*="cardNumber"]',
    '.StripeElement', '#card-element', '#paypal-button-container',
    '.apple-pay-button', '.gpay-button', '[data-braintree-id]',
  ];

  const PRICE_REGEX = /\$\s*([\d,]+\.?\d{0,2})/;

  const MERCHANT_DOMAINS = {
    'amazon.com': { name: 'Amazon', category: 'online_shopping' },
    'walmart.com': { name: 'Walmart', category: 'department_store' },
    'target.com': { name: 'Target', category: 'department_store' },
    'bestbuy.com': { name: 'Best Buy', category: 'online_shopping' },
    'ebay.com': { name: 'eBay', category: 'online_shopping' },
    'etsy.com': { name: 'Etsy', category: 'online_shopping' },
    'apple.com': { name: 'Apple', category: 'online_shopping' },
    'nike.com': { name: 'Nike', category: 'online_shopping' },
    'costco.com': { name: 'Costco', category: 'groceries' },
    'homedepot.com': { name: 'Home Depot', category: 'home_improvement' },
    'lowes.com': { name: "Lowe's", category: 'home_improvement' },
    'wayfair.com': { name: 'Wayfair', category: 'online_shopping' },
    'nordstrom.com': { name: 'Nordstrom', category: 'department_store' },
    'macys.com': { name: "Macy's", category: 'department_store' },
    'kohls.com': { name: "Kohl's", category: 'department_store' },
    'adidas.com': { name: 'Adidas', category: 'online_shopping' },
    'lululemon.com': { name: 'Lululemon', category: 'online_shopping' },
    'instacart.com': { name: 'Instacart', category: 'groceries' },
    'doordash.com': { name: 'DoorDash', category: 'dining' },
    'ubereats.com': { name: 'Uber Eats', category: 'dining' },
    'grubhub.com': { name: 'Grubhub', category: 'dining' },
    'expedia.com': { name: 'Expedia', category: 'travel' },
    'booking.com': { name: 'Booking.com', category: 'travel' },
    'airbnb.com': { name: 'Airbnb', category: 'travel' },
    'hotels.com': { name: 'Hotels.com', category: 'travel' },
    'southwest.com': { name: 'Southwest Airlines', category: 'travel' },
    'delta.com': { name: 'Delta Airlines', category: 'travel' },
    'united.com': { name: 'United Airlines', category: 'travel' },
    'aa.com': { name: 'American Airlines', category: 'travel' },
    'netflix.com': { name: 'Netflix', category: 'streaming' },
    'spotify.com': { name: 'Spotify', category: 'streaming' },
    'disneyplus.com': { name: 'Disney+', category: 'streaming' },
    'hulu.com': { name: 'Hulu', category: 'streaming' },
    'uber.com': { name: 'Uber', category: 'transit' },
    'lyft.com': { name: 'Lyft', category: 'transit' },
    'cvs.com': { name: 'CVS', category: 'drugstores' },
    'walgreens.com': { name: 'Walgreens', category: 'drugstores' },
    'sephora.com': { name: 'Sephora', category: 'online_shopping' },
    'ikea.com': { name: 'IKEA', category: 'home_improvement' },
    'wholefoodsmarket.com': { name: 'Whole Foods', category: 'groceries' },
    'kroger.com': { name: 'Kroger', category: 'groceries' },
    'safeway.com': { name: 'Safeway', category: 'groceries' },
    'albertsons.com': { name: 'Albertsons', category: 'groceries' },
    'publix.com': { name: 'Publix', category: 'groceries' },
    'traderjoes.com': { name: "Trader Joe's", category: 'groceries' },
    'aldi.us': { name: 'ALDI', category: 'groceries' },
    'wegmans.com': { name: 'Wegmans', category: 'groceries' },
    'sprouts.com': { name: 'Sprouts', category: 'groceries' },
    'freshdirect.com': { name: 'FreshDirect', category: 'groceries' },
    'shipt.com': { name: 'Shipt', category: 'groceries' },
    'heb.com': { name: 'H-E-B', category: 'groceries' },
    'gopuff.com': { name: 'Gopuff', category: 'groceries' },
  };

  const CATEGORY_LABELS = {
    dining: 'Dining', groceries: 'Groceries', gas: 'Gas', travel: 'Travel',
    entertainment: 'Entertainment', online_shopping: 'Online Shopping',
    streaming: 'Streaming', drugstores: 'Drugstores',
    home_improvement: 'Home Improvement', transit: 'Transit',
    department_store: 'Department Store', other: 'Other',
  };

  // ── Message Listener ──

  chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    if (message.type === 'CHECK_PAGE') {
      handleCheckPage(message);
      sendResponse({ received: true });
    }
    if (message.type === 'SHOW_OVERLAY') {
      showOverlay(message);
      sendResponse({ received: true });
    }
    if (message.type === 'HIDE_OVERLAY') {
      hideOverlay();
      sendResponse({ received: true });
    }
    if (message.type === 'GET_PAGE_AMOUNT') {
      const amount = extractAmountFromPage();
      sendResponse({ amount: amount });
    }
    return true;
  });

  // ── Page Analysis ──

  function handleCheckPage(message) {
    const url = message.url || window.location.href;

    // Step 1: Detect merchant from URL
    const merchant = detectMerchantFromUrl(url);

    // Step 2: Check if this looks like a checkout page
    const checkoutScore = getCheckoutScore(url);

    // Step 3: If on a known merchant with checkout signals, request recommendation
    if (merchant && checkoutScore >= 50) {
      const amount = extractAmountFromPage();

      chrome.runtime.sendMessage({
        type: 'CHECKOUT_DETECTED',
        merchant: merchant.name,
        category: merchant.category,
        amount: amount || null,
        url: url,
        isCheckout: true,
      });
    }
    // Even if not checkout, if on known merchant and score >= 30, still notify
    else if (merchant && checkoutScore >= 30) {
      const amount = extractAmountFromPage();
      chrome.runtime.sendMessage({
        type: 'CHECKOUT_DETECTED',
        merchant: merchant.name,
        category: merchant.category,
        amount: amount || null,
        url: url,
        isCheckout: true,
      });
    }
    // On a known merchant but not checkout — notify for proactive recommendation
    else if (merchant) {
      chrome.runtime.sendMessage({
        type: 'MERCHANT_PAGE_DETECTED',
        merchant: merchant.name,
        category: merchant.category,
        url: url,
      });
    }
    // Unknown site — try smart detection from page signals
    else {
      const unknown = detectUnknownMerchant();
      if (unknown && unknown.confidence >= 25) {
        const checkoutScore = getCheckoutScore(url);
        if (checkoutScore >= 30) {
          const amount = extractAmountFromPage();
          chrome.runtime.sendMessage({
            type: 'CHECKOUT_DETECTED',
            merchant: unknown.name,
            category: unknown.category,
            amount: amount || null,
            url: url,
            isCheckout: true,
            isDiscovered: true,
          });
        } else {
          chrome.runtime.sendMessage({
            type: 'UNKNOWN_MERCHANT_DETECTED',
            merchant: unknown.name,
            category: unknown.category,
            confidence: unknown.confidence,
            url: url,
            domain: window.location.hostname.replace(/^www\./, '').toLowerCase(),
          });
        }
      }
    }
  }

  function detectMerchantFromUrl(url) {
    try {
      let hostname = new URL(url).hostname.toLowerCase();
      if (hostname.startsWith('www.')) hostname = hostname.slice(4);

      for (const [domain, info] of Object.entries(MERCHANT_DOMAINS)) {
        if (hostname === domain || hostname.endsWith('.' + domain)) {
          return info;
        }
      }

      if (cachedServerMerchants) {
        for (const [domain, info] of Object.entries(cachedServerMerchants)) {
          if (hostname === domain || hostname.endsWith('.' + domain)) {
            return info;
          }
        }
      }
    } catch {}
    return null;
  }

  function detectUnknownMerchant() {
    try {
      const hostname = window.location.hostname.replace(/^www\./, '').toLowerCase();

      if (/google\.|facebook\.|twitter\.|reddit\.|youtube\.|wikipedia\.|github\.|stackoverflow\.|linkedin\.|instagram\.|tiktok\.|x\.com|whatsapp\.|telegram\.|discord\.|slack\.|zoom\.|microsoft\.|apple\.com\/(?!shop)/.test(hostname)) {
        return null;
      }

      let merchantName = null;
      let confidence = 0;
      let detectedCategory = 'online_shopping';

      const jsonLdScripts = document.querySelectorAll('script[type="application/ld+json"]');
      for (const script of jsonLdScripts) {
        try {
          const data = JSON.parse(script.textContent);
          const items = Array.isArray(data) ? data : [data];
          for (const item of items) {
            const t = (item['@type'] || '').toLowerCase();
            if (t === 'product' || t === 'offer' || t === 'itempage') {
              confidence += 40;
            }
            if (t === 'restaurant' || t === 'foodestablishment') {
              confidence += 40;
              detectedCategory = 'dining';
            }
            if (t === 'hotel' || t === 'lodgingbusiness') {
              confidence += 40;
              detectedCategory = 'travel';
            }
            if (item.name && t === 'organization') {
              merchantName = item.name;
            }
            if (item.brand?.name && !merchantName) {
              merchantName = item.brand.name;
            }
          }
        } catch {}
      }

      const ogSiteName = document.querySelector('meta[property="og:site_name"]');
      if (ogSiteName?.content) {
        merchantName = merchantName || ogSiteName.content;
        confidence += 5;
      }

      const ogType = document.querySelector('meta[property="og:type"]');
      if (ogType?.content?.includes('product')) {
        confidence += 20;
      }

      if (window.Shopify || document.querySelector('meta[name="shopify-checkout-api-token"]') || document.querySelector('link[href*="cdn.shopify.com"]')) {
        confidence += 35;
      }
      if (document.querySelector('.woocommerce, .woocommerce-page, meta[name="generator"][content*="WooCommerce"]')) {
        confidence += 35;
      }
      if (document.querySelector('script[src*="bigcommerce"], meta[name="platform"][content*="BigCommerce"]')) {
        confidence += 35;
      }
      if (document.querySelector('meta[name="generator"][content*="Magento"]')) {
        confidence += 35;
      }
      if (document.querySelector('.sqs-block-commerce, .squarespace-commerce')) {
        confidence += 30;
      }

      const cartSignals = [
        'button', 'a', 'input[type="submit"]', '[role="button"]'
      ];
      for (const sel of cartSignals) {
        const els = document.querySelectorAll(sel);
        for (const el of els) {
          const text = (el.textContent || el.value || '').toLowerCase().trim();
          if (/add to (cart|bag|basket)|buy now|shop now|add to wishlist/i.test(text)) {
            confidence += 15;
            break;
          }
        }
      }

      if (document.querySelector('[class*="product-price"], [class*="productPrice"], [data-price], [class*="price-tag"], [itemprop="price"]')) {
        confidence += 10;
      }

      if (document.querySelector('a[href*="/cart"], a[href*="/basket"], a[href*="/bag"], [class*="cart-icon"], [class*="shopping-cart"], [class*="minicart"]')) {
        confidence += 10;
      }

      if (confidence < 25) return null;

      if (!merchantName) {
        const titleEl = document.querySelector('title');
        if (titleEl?.textContent) {
          let title = titleEl.textContent.trim();
          title = title.split(/\s*[|\-–—:]\s*/)[0].trim();
          if (title.length > 1 && title.length < 50) {
            merchantName = title;
          }
        }
      }

      if (!merchantName) {
        const parts = hostname.split('.');
        const domain = parts.length >= 2 ? parts[parts.length - 2] : parts[0];
        merchantName = domain.charAt(0).toUpperCase() + domain.slice(1);
      }

      return { name: merchantName, category: detectedCategory, confidence };
    } catch {}
    return null;
  }

  function getCheckoutScore(url) {
    let score = 0;

    // URL patterns
    for (const pattern of CHECKOUT_URL_PATTERNS) {
      if (pattern.test(url)) { score += 40; break; }
    }

    // Card input fields
    for (const selector of CARD_INPUT_SELECTORS) {
      try {
        if (document.querySelector(selector)) { score += 30; break; }
      } catch {}
    }

    // Payment buttons
    const buttons = document.querySelectorAll('button, input[type="submit"], a[role="button"]');
    for (const btn of buttons) {
      const text = (btn.textContent || btn.value || '').toLowerCase().trim();
      if (PAYMENT_BUTTON_TEXTS.some(p => text.includes(p))) {
        score += 20; break;
      }
    }

    // Known merchant gives 10 bonus
    if (detectMerchantFromUrl(window.location.href)) score += 10;

    return score;
  }

  function extractAmountFromPage() {
    const selectors = [
      '[data-testid*="total"]', '[data-testid*="Total"]',
      '.order-total', '.cart-total', '.grand-total',
      '.order-summary-total', '.total-price',
      '#order-total', '#cart-total',
      '[class*="orderTotal"]', '[class*="cartTotal"]', '[class*="grandTotal"]',
      '[class*="order-total"]', '[class*="cart-total"]', '[class*="grand-total"]',
      '[class*="OrderTotal"]', '[class*="CartTotal"]', '[class*="GrandTotal"]',
      '.summary__total', '[class*="summary-total"]', '[class*="summaryTotal"]',
      '.price--total', '[class*="priceTotal"]',
      '#orderSummary [class*="total"]',
      '.checkout-total', '[class*="checkoutTotal"]',
      '[data-automation*="total"]', '[data-component*="total"]',
    ];

    for (const selector of selectors) {
      try {
        const els = document.querySelectorAll(selector);
        for (const el of els) {
          const text = (el.textContent || '').trim();
          if (text.length > 200) continue;
          const match = text.match(PRICE_REGEX);
          if (match) {
            const val = parseFloat(match[1].replace(/,/g, ''));
            if (val > 0 && val < 1000000) return val;
          }
        }
      } catch {}
    }

    const totalKeywords = [
      /order\s*total/i, /grand\s*total/i, /estimated\s*total/i,
      /cart\s*total/i, /your\s*total/i, /amount\s*due/i,
      /total\s*due/i, /total\s*:/i, /pay\s*total/i,
      /checkout\s*total/i, /purchase\s*total/i,
    ];

    const scanEls = document.querySelectorAll(
      'td, th, span, div, p, strong, b, label, li, dt, dd, h2, h3, h4, tr'
    );

    for (const el of scanEls) {
      const text = (el.textContent || '').trim();
      if (text.length > 300 || text.length < 3) continue;

      for (const kw of totalKeywords) {
        if (kw.test(text)) {
          const match = text.match(PRICE_REGEX);
          if (match) {
            const val = parseFloat(match[1].replace(/,/g, ''));
            if (val > 0 && val < 1000000) return val;
          }

          const nextEl = el.nextElementSibling;
          if (nextEl) {
            const nextMatch = (nextEl.textContent || '').match(PRICE_REGEX);
            if (nextMatch) {
              const val = parseFloat(nextMatch[1].replace(/,/g, ''));
              if (val > 0 && val < 1000000) return val;
            }
          }

          const parent = el.parentElement;
          if (parent) {
            const parentMatch = (parent.textContent || '').match(PRICE_REGEX);
            if (parentMatch) {
              const val = parseFloat(parentMatch[1].replace(/,/g, ''));
              if (val > 0 && val < 1000000) return val;
            }
          }
        }
      }
    }

    const allPrices = [];
    const priceEls = document.querySelectorAll('span, div, td, strong, b, p');
    for (const el of priceEls) {
      const text = (el.textContent || '').trim();
      if (text.length > 50 || text.length < 2) continue;
      const children = el.children.length;
      if (children > 3) continue;
      const match = text.match(PRICE_REGEX);
      if (match) {
        const val = parseFloat(match[1].replace(/,/g, ''));
        if (val > 1 && val < 1000000) allPrices.push(val);
      }
    }
    if (allPrices.length > 0) {
      return Math.max(...allPrices);
    }

    return null;
  }

  // ── Overlay Rendering ──

  function escapeHtml(str) {
    return String(str)
      .replace(/&/g, '&amp;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;')
      .replace(/"/g, '&quot;')
      .replace(/'/g, '&#39;');
  }

  function showOverlay(data) {
    hideOverlay(); // Remove existing

    const { recommendation, allCards, merchant, category, amount, settings } = data;
    if (!recommendation) return;

    currentRecommendation = data;
    const card = recommendation.card || recommendation;
    const position = settings?.overlayPosition || 'bottom-right';
    const autoHide = settings?.autoHideSeconds || 15;
    const showSecond = settings?.showSecondBest !== false;

    // Build overlay HTML
    const overlay = document.createElement('div');
    overlay.id = 'superpay-overlay';
    overlay.className = `superpay-overlay superpay-${position}`;

    const rewardLabel = card.rewardType === 'miles'
      ? `${Math.round(recommendation.rewardsEarned || 0)} miles`
      : `$${(recommendation.rewardsEarned || 0).toFixed(2)}`;

    const categoryLabel = CATEGORY_LABELS[category] || escapeHtml(category);
    const cardName = card.nickname || card.issuer || 'Your Card';
    const lastFour = card.lastFour ? ` ····${card.lastFour}` : '';

    let secondBestHtml = '';
    if (showSecond && allCards && allCards.length > 1) {
      const second = allCards[1];
      const sCard = second.card || second;
      const sLabel = sCard.rewardType === 'miles'
        ? `${Math.round(second.rewardsEarned)} mi`
        : `$${second.rewardsEarned.toFixed(2)}`;
      secondBestHtml = `
        <div class="superpay-second-best">
          2nd: ${escapeHtml(sCard.nickname || sCard.issuer)} (${second.rewardRate}x · ${sLabel})
        </div>
      `;
    }

    overlay.innerHTML = `
      <div class="superpay-overlay-header">
        <div class="superpay-logo">
          <img src="${chrome.runtime.getURL('icons/icon32.png')}" alt="SuperPay" class="superpay-logo-icon">
          <span class="superpay-logo-text">SuperPay</span>
        </div>
        <div class="superpay-header-actions">
          <button class="superpay-btn-settings" title="Settings">${ICON_SETTINGS}</button>
          <button class="superpay-btn-close" title="Dismiss">&times;</button>
        </div>
      </div>
      <div class="superpay-overlay-body">
        <div class="superpay-merchant-info">
          <span class="superpay-merchant-name">Shopping at ${escapeHtml(merchant)}</span>
          <span class="superpay-merchant-category">${categoryLabel}</span>
        </div>
        <div class="superpay-use-label">USE THIS CARD</div>
        <div class="superpay-card-rec">
          <div class="superpay-card-rec-left">
            <div class="superpay-card-name">${escapeHtml(cardName)}${escapeHtml(lastFour)}</div>
            <div class="superpay-card-details">
              ${recommendation.rewardRate}x ${card.rewardType === 'cashback' ? 'cash back' : card.rewardType}
              ${formatValueLabel(recommendation.rewardRate, card.rewardType, amount) ? `<span class="superpay-value-context">${formatValueLabel(recommendation.rewardRate, card.rewardType, amount)}</span>` : ''}
              <span class="superpay-reward-amount">· ~${rewardLabel}${amount ? ` on $${amount.toFixed(2)}` : ''}</span>
            </div>
          </div>
          <div class="superpay-card-rec-badge">Best</div>
        </div>
        ${secondBestHtml}
        <div class="superpay-ai-section" id="superpay-ai-section" style="display:none">
          <div class="superpay-ai-label">${ICON_AI} AI Insight</div>
          <div class="superpay-ai-text" id="superpay-ai-text"></div>
        </div>
        <div class="superpay-goals-section" id="superpay-goals-section" style="display:none">
          <button class="superpay-goals-toggle" id="superpay-goals-toggle">
            <span class="superpay-goals-toggle-label">BEST CARD BY GOAL</span>
            <span class="superpay-goals-arrow" id="superpay-goals-arrow">&#9660;</span>
          </button>
          <div class="superpay-goals-list" id="superpay-goals-list" style="display:none"></div>
        </div>
      </div>
      <div class="superpay-overlay-footer">
        <button class="superpay-btn-log" id="superpay-log-btn">${ICON_CHECK} Log this purchase</button>
        <a class="superpay-btn-app" href="#" id="superpay-open-app">Open app</a>
      </div>
    `;

    document.body.appendChild(overlay);
    overlayElement = overlay;

    // Slide in animation
    requestAnimationFrame(() => {
      overlay.classList.add('superpay-visible');
    });

    // Event listeners
    overlay.querySelector('.superpay-btn-close').addEventListener('click', hideOverlay);
    overlay.querySelector('.superpay-btn-settings').addEventListener('click', () => {
      chrome.runtime.sendMessage({ type: 'OPEN_OPTIONS' });
    });

    const logBtn = overlay.querySelector('#superpay-log-btn');
    logBtn.addEventListener('click', () => {
      logBtn.innerHTML = `${ICON_CHECK} Logged!`;
      logBtn.disabled = true;
      logBtn.classList.add('superpay-logged');
      chrome.runtime.sendMessage({
        type: 'LOG_TRANSACTION',
        transaction: {
          amount: amount || 100,
          merchant: merchant,
          category: category,
          date: new Date().toISOString().split('T')[0],
          cardId: card.id || card.cardId,
          rewardsEarned: recommendation.rewardsEarned || 0,
          wasOptimal: true,
          optimalCardId: card.id || card.cardId,
          optimalRewards: recommendation.rewardsEarned || 0,
        },
      });
    });

    // Fetch AI explanation (async, non-blocking)
    if (settings?.showAiExplanation !== false) {
      fetchAiExplanation(data);
    }

    // Fetch multi-goal recommendations (async)
    fetchMultiGoalForOverlay(data);

    // Auto-hide
    if (autoHide > 0) {
      overlayTimeout = setTimeout(hideOverlay, autoHide * 1000);
    }

    // Pause auto-hide on hover
    overlay.addEventListener('mouseenter', () => {
      if (overlayTimeout) clearTimeout(overlayTimeout);
    });
    overlay.addEventListener('mouseleave', () => {
      if (autoHide > 0) {
        overlayTimeout = setTimeout(hideOverlay, 5000);
      }
    });
  }

  async function fetchAiExplanation(data) {
    const { recommendation, allCards, merchant, category, amount, settings } = data;
    const card = recommendation.card || recommendation;
    const autoHide = settings?.autoHideSeconds || 15;

    const aiSection = document.getElementById('superpay-ai-section');
    const aiText = document.getElementById('superpay-ai-text');
    if (aiSection && aiText) {
      aiSection.style.display = 'block';
      aiText.innerHTML = '<div class="superpay-ai-shimmer"><div class="superpay-shimmer-line"></div><div class="superpay-shimmer-line superpay-shimmer-short"></div></div>';
    }

    try {
      const stored = await chrome.storage.local.get(['goals']);
      const userGoals = stored.goals || [];

      const response = await chrome.runtime.sendMessage({
        type: 'GET_AI_EXPLANATION',
        data: {
          bestCard: {
            nickname: card.nickname || card.issuer,
            rewardRate: recommendation.rewardRate,
            rewardType: card.rewardType,
            rewardsEarned: recommendation.rewardsEarned,
            utilization: card.creditLimit
              ? ((card.currentBalance || 0) / card.creditLimit) * 100
              : null,
          },
          otherCards: (allCards || []).slice(1, 4).map(c => ({
            nickname: (c.card || c).nickname || (c.card || c).issuer,
            rewardRate: c.rewardRate,
            rewardType: (c.card || c).rewardType,
            rewardsEarned: c.rewardsEarned,
          })),
          merchant,
          category,
          amount: amount || null,
          goals: userGoals,
        },
      });

      if (response?.explanation) {
        if (aiSection && aiText) {
          aiText.textContent = response.explanation;
          aiSection.style.display = 'block';

          if (overlayTimeout) clearTimeout(overlayTimeout);
          if (autoHide > 0) {
            overlayTimeout = setTimeout(hideOverlay, autoHide * 1000);
          }
        }
      } else {
        if (aiSection) aiSection.style.display = 'none';
      }
    } catch {
      if (aiSection) aiSection.style.display = 'none';
    }
  }

  async function fetchMultiGoalForOverlay(data) {
    const { category, amount } = data;
    const section = document.getElementById('superpay-goals-section');
    const list = document.getElementById('superpay-goals-list');
    const toggle = document.getElementById('superpay-goals-toggle');
    const arrow = document.getElementById('superpay-goals-arrow');
    if (!section || !list || !toggle) return;

    try {
      const response = await chrome.runtime.sendMessage({
        type: 'GET_MULTI_GOAL_RECOMMENDATION',
        category,
        amount: amount || null,
      });

      if (!response?.results || response.results.length === 0) return;

      section.style.display = 'block';

      list.innerHTML = response.results.map(r => {
        const safeRate = parseFloat(r.rewardRate);
        const rate = isFinite(safeRate) ? safeRate : 0;
        const rewardLabel = r.rewardType === 'miles'
          ? `${rate}x mi`
          : r.rewardType === 'points'
            ? `${rate}x pts`
            : `${rate}x cb`;
        const valLabel = formatValueLabel(rate, r.rewardType, amount);
        const valHtml = valLabel ? `<span class="superpay-mg-value">${valLabel}</span>` : '';
        const activeClass = r.isActive ? 'superpay-mg-active' : '';
        const dot = r.isActive ? '<span class="superpay-mg-dot"></span>' : '';
        const safeColor = escapeHtml(r.color);
        const goalSvg = GOAL_SVG_MAP[r.goalId] ? GOAL_SVG_MAP[r.goalId](safeColor) : svgIcon('<circle cx="12" cy="12" r="10"/>', safeColor);
        return `<div class="superpay-mg-row ${activeClass}">
          <span class="superpay-mg-icon">${goalSvg}</span>
          <span class="superpay-mg-label" style="color:${safeColor}">${escapeHtml(r.label)}</span>
          ${dot}
          <span class="superpay-mg-card">${escapeHtml(r.cardName)}</span>
          <span class="superpay-mg-rate">${rewardLabel}</span>
          ${valHtml}
        </div>`;
      }).join('');

      const newToggle = toggle.cloneNode(true);
      toggle.parentNode.replaceChild(newToggle, toggle);
      let expanded = false;
      newToggle.addEventListener('click', () => {
        expanded = !expanded;
        list.style.display = expanded ? 'block' : 'none';
        const arrowEl = newToggle.querySelector('.superpay-goals-arrow');
        if (arrowEl) arrowEl.innerHTML = expanded ? '&#9650;' : '&#9660;';
      });

      const autoHide = data.settings?.autoHideSeconds || 15;
      if (overlayTimeout) clearTimeout(overlayTimeout);
      if (autoHide > 0) {
        overlayTimeout = setTimeout(hideOverlay, autoHide * 1000);
      }
    } catch {}
  }

  function hideOverlay() {
    if (overlayTimeout) clearTimeout(overlayTimeout);
    if (overlayElement) {
      overlayElement.classList.remove('superpay-visible');
      overlayElement.classList.add('superpay-hiding');
      setTimeout(() => {
        overlayElement?.remove();
        overlayElement = null;
      }, 300);
    }
  }

  // ── Initial Page Check ──
  // Run checkout detection on page load
  setTimeout(() => {
    handleCheckPage({ url: window.location.href });
  }, 1500); // Wait for page to fully render

})();