// lib/api.js — SuperPay API Client
// All communication with the SuperPay backend.
// Used by popup.js for direct API calls outside the service worker context.

const API_BASE = 'https://superpayrewards.com';

class SuperPayAPI {
  constructor() {
    this.token = null;
  }

  async init() {
    const data = await chrome.storage.local.get(['authToken']);
    this.token = data.authToken;
  }

  headers() {
    const h = { 'Content-Type': 'application/json' };
    if (this.token) h['Authorization'] = `Bearer ${this.token}`;
    return h;
  }

  // ── Auth ──

  async login(email, password) {
    const res = await fetch(`${API_BASE}/api/auth/login`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ email, password }),
    });
    const data = await res.json();
    if (!res.ok) throw new Error(data.message || 'Login failed');
    this.token = data.token;
    return data;
  }

  async signup(email, password, name) {
    const res = await fetch(`${API_BASE}/api/auth/signup`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ email, password, name }),
    });
    const data = await res.json();
    if (!res.ok) throw new Error(data.message || 'Signup failed');
    this.token = data.token;
    return data;
  }

  async getMe() {
    const res = await fetch(`${API_BASE}/api/auth/me`, {
      headers: this.headers(),
    });
    if (!res.ok) throw new Error('Not authenticated');
    return await res.json();
  }

  // ── Cards ──

  async getCards() {
    const res = await fetch(`${API_BASE}/api/cards`, {
      headers: this.headers(),
    });
    if (!res.ok) throw new Error('Failed to fetch cards');
    return await res.json();
  }

  // ── Checkout URL Parsing ──

  async parseCheckoutUrl(url) {
    const res = await fetch(`${API_BASE}/api/parse-checkout-url`, {
      method: 'POST',
      headers: this.headers(),
      body: JSON.stringify({ url }),
    });
    if (!res.ok) return { found: false };
    return await res.json();
  }

  // ── AI Explanation ──

  async getExplanation({ bestCard, otherCards, merchant, category, amount, goals }) {
    const res = await fetch(`${API_BASE}/api/explain-recommendation`, {
      method: 'POST',
      headers: this.headers(),
      body: JSON.stringify({ bestCard, otherCards, merchant, category, amount, goals }),
    });
    if (!res.ok) return { explanation: '' };
    return await res.json();
  }

  // ── Extension Config (merchant sync) ──

  async getExtensionConfig() {
    const res = await fetch(`${API_BASE}/api/extension/config`, {
      headers: this.headers(),
    });
    if (!res.ok) throw new Error('Failed to fetch config');
    return await res.json();
  }

  // ── Transactions ──

  async logTransaction(transaction) {
    const res = await fetch(`${API_BASE}/api/transactions`, {
      method: 'POST',
      headers: this.headers(),
      body: JSON.stringify(transaction),
    });
    if (!res.ok) throw new Error('Failed to log transaction');
    return await res.json();
  }
}

// Export for use in popup.js (loaded via <script>)
window.SuperPayAPI = SuperPayAPI;