// lib/checkout-detector.js — Checkout Page Detection
// Uses multiple signals to determine if the user is on a checkout page.
// Returns a confidence score — overlay shows at 50+ points.

const CHECKOUT_URL_PATTERNS = [
  /\/checkout/i,
  /\/cart/i,
  /\/basket/i,
  /\/order/i,
  /\/payment/i,
  /\/pay\b/i,
  /\/billing/i,
  /\/purchase/i,
  /\/subscribe/i,
  /\/gp\/buy/i,           // Amazon
  /\/chkout/i,            // eBay
  /\/secure/i,
  /\/place-order/i,
  /\/confirm-order/i,
];

const PAYMENT_BUTTON_TEXTS = [
  'place order', 'place your order', 'complete purchase', 'complete order',
  'pay now', 'pay with', 'submit order', 'confirm order', 'confirm purchase',
  'buy now', 'proceed to checkout', 'continue to payment', 'review order',
  'finalize order', 'process payment', 'submit payment', 'checkout',
  'purchase', 'subscribe', 'start trial', 'join now',
];

const CARD_INPUT_SELECTORS = [
  'input[autocomplete="cc-number"]',
  'input[autocomplete="cc-name"]',
  'input[autocomplete="cc-exp"]',
  'input[autocomplete="cc-csc"]',
  'input[name*="card"]',
  'input[name*="credit"]',
  'input[name*="cc-"]',
  'input[name*="ccnum"]',
  'input[id*="card-number"]',
  'input[id*="cardNumber"]',
  'input[data-testid*="card"]',
  'input[placeholder*="card number"]',
  'input[placeholder*="Card number"]',
  '[data-braintree-id]',     // Braintree payment
  '.StripeElement',           // Stripe Elements
  '#card-element',            // Stripe
  '[data-recurly]',           // Recurly
  '.adyen-checkout',          // Adyen
  '#paypal-button-container', // PayPal
  '.apple-pay-button',        // Apple Pay
  '.gpay-button',             // Google Pay
];

const PRICE_SELECTORS = [
  '[data-testid*="total"]',
  '[data-testid*="price"]',
  '.order-total', '.cart-total', '.grand-total',
  '.order-summary-total', '.cart-summary-total',
  '#order-total', '#cart-total',
  '[class*="orderTotal"]', '[class*="cartTotal"]',
  '[class*="grandTotal"]', '[class*="totalPrice"]',
  '.summary-total', '.checkout-total',
];

function detectCheckout(url, document) {
  let score = 0;
  const signals = [];

  // Signal 1: URL pattern match (40 points)
  for (const pattern of CHECKOUT_URL_PATTERNS) {
    if (pattern.test(url)) {
      score += 40;
      signals.push('URL matches checkout pattern');
      break;
    }
  }

  // Signal 2: Card input fields found (30 points)
  for (const selector of CARD_INPUT_SELECTORS) {
    try {
      if (document.querySelector(selector)) {
        score += 30;
        signals.push('Payment input field detected');
        break;
      }
    } catch {}
  }

  // Signal 3: Payment button found (20 points)
  const buttons = document.querySelectorAll('button, input[type="submit"], a[role="button"]');
  for (const btn of buttons) {
    const text = (btn.textContent || btn.value || '').toLowerCase().trim();
    for (const pattern of PAYMENT_BUTTON_TEXTS) {
      if (text.includes(pattern)) {
        score += 20;
        signals.push(`Payment button found: "${text.slice(0, 40)}"`);
        break;
      }
    }
    if (score >= 20) break; // Only count once
  }

  // Signal 4: Known merchant domain (10 points)
  // This is checked separately using MERCHANT_DOMAINS

  return {
    isCheckout: score >= 50,
    score,
    signals,
  };
}

if (typeof window !== 'undefined') {
  window.detectCheckout = detectCheckout;
  window.PRICE_SELECTORS = PRICE_SELECTORS;
}