// options.js — SuperPay Extension Settings

document.addEventListener('DOMContentLoaded', async () => {
  const overlayEnabled = document.getElementById('overlay-enabled');
  const showAi = document.getElementById('show-ai');
  const showSecond = document.getElementById('show-second');
  const overlayPosition = document.getElementById('overlay-position');
  const autoHide = document.getElementById('auto-hide');
  const blacklistInput = document.getElementById('blacklist-input');
  const addBlacklistBtn = document.getElementById('add-blacklist');
  const blacklistItems = document.getElementById('blacklist-items');
  const accountEmail = document.getElementById('account-email');
  const accountStatus = document.getElementById('account-status');
  const saveStatus = document.getElementById('save-status');

  // Load current settings
  const data = await chrome.storage.local.get(['settings', 'user']);

  const settings = data.settings || {
    overlayEnabled: true,
    overlayPosition: 'bottom-right',
    autoHideSeconds: 15,
    showSecondBest: true,
    showAiExplanation: true,
    blacklistedDomains: [],
  };

  // Populate form
  overlayEnabled.checked = settings.overlayEnabled;
  showAi.checked = settings.showAiExplanation;
  showSecond.checked = settings.showSecondBest;
  overlayPosition.value = settings.overlayPosition;
  autoHide.value = settings.autoHideSeconds;
  renderBlacklist(settings.blacklistedDomains || []);

  if (data.user) {
    accountEmail.textContent = data.user.email;
    accountStatus.textContent = `Logged in as ${data.user.name || data.user.email}`;
  }

  // ── Auto-save on change ──

  function saveSettings() {
    const updated = {
      overlayEnabled: overlayEnabled.checked,
      overlayPosition: overlayPosition.value,
      autoHideSeconds: parseInt(autoHide.value) || 15,
      showSecondBest: showSecond.checked,
      showAiExplanation: showAi.checked,
      blacklistedDomains: settings.blacklistedDomains || [],
    };

    chrome.storage.local.set({ settings: updated });
    settings.overlayEnabled = updated.overlayEnabled;
    settings.overlayPosition = updated.overlayPosition;
    settings.autoHideSeconds = updated.autoHideSeconds;
    settings.showSecondBest = updated.showSecondBest;
    settings.showAiExplanation = updated.showAiExplanation;

    showSaveStatus('Settings saved');
  }

  overlayEnabled.addEventListener('change', saveSettings);
  showAi.addEventListener('change', saveSettings);
  showSecond.addEventListener('change', saveSettings);
  overlayPosition.addEventListener('change', saveSettings);
  autoHide.addEventListener('change', saveSettings);

  // ── Blacklist ──

  addBlacklistBtn.addEventListener('click', () => {
    const domain = blacklistInput.value.trim().toLowerCase().replace(/^(https?:\/\/)?(www\.)?/, '');
    if (!domain) return;

    if (!settings.blacklistedDomains) settings.blacklistedDomains = [];
    if (settings.blacklistedDomains.includes(domain)) return;

    settings.blacklistedDomains.push(domain);
    renderBlacklist(settings.blacklistedDomains);
    blacklistInput.value = '';
    saveSettings();
  });

  blacklistInput.addEventListener('keydown', (e) => {
    if (e.key === 'Enter') addBlacklistBtn.click();
  });

  function renderBlacklist(domains) {
    blacklistItems.innerHTML = '';
    domains.forEach(domain => {
      const item = document.createElement('div');
      item.className = 'blacklist-item';

      const span = document.createElement('span');
      span.className = 'blacklist-domain';
      span.textContent = domain;

      const btn = document.createElement('button');
      btn.className = 'blacklist-remove';
      btn.textContent = '\u00D7';
      btn.dataset.domain = domain;
      btn.addEventListener('click', () => {
        settings.blacklistedDomains = settings.blacklistedDomains.filter(x => x !== domain);
        renderBlacklist(settings.blacklistedDomains);
        saveSettings();
      });

      item.appendChild(span);
      item.appendChild(btn);
      blacklistItems.appendChild(item);
    });
  }

  function showSaveStatus(text) {
    saveStatus.textContent = text;
    saveStatus.classList.add('visible');
    setTimeout(() => saveStatus.classList.remove('visible'), 2000);
  }
});